/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.io.IOException;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.opensaml.profile.context.ProfileRequestContext;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

import net.shibboleth.ext.spring.resource.PreferFileSystemResourceLoader;
import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.oidc.metadata.policy.MetadataPolicyResolver;
import net.shibboleth.oidc.metadata.policy.impl.OIDCMetadataPolicyResolver;
import net.shibboleth.oidc.profile.config.navigate.ResolverBasedRegistrationMetadataPolicyLookupFunction;
import net.shibboleth.utilities.java.support.annotation.ParameterName;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A factory for building a function that resolves maps of metadata policies from the profile request context. The
 * function uses a newly created {@link MetadataPolicyResolver}. The resolver is backed by a newly instantiated
 * {@link BatchMetadataCache}. Every invocation of this factory will therefore build a new resolver and cache instance
 * from the supplied cache specification.
 */
@ThreadSafe
public class MetadataPolicyLookupStrategyFactory {
    

    /**
     * Build a Function that maps a profile request context to a map of metadata policies. The map is resolved
     * from a newly instantiated {@link MetadataPolicyResolver}. The Resolver calls a new {@link BatchMetadataCache}
     * which is hard-wired to use a default file loading strategy with the file resource supplied.
     * 
     * @param resource the file to inject into the loading strategy. Can be null or empty.
     * @param cacheSpec the metadata cache specification
     * @param criteriaSetLookupStrategy the lookup strategy for the criteria set used for the metadata policy resolver.
     * @param id the identifier/name for the back-end cache created
     * 
     * @return the function
     * 
     * @throws ComponentInitializationException on error.
     * @throws IOException on error.
     */
    public Function<ProfileRequestContext,Map<String,MetadataPolicy>> buildFileLoadingMetadataPolicyResolver(
            @ParameterName(name="resource") @Nullable final String resource,
            @ParameterName(name="cacheSpec") @Nonnull 
            final BatchMetadataCacheBuilderSpec<String, Map<String, MetadataPolicy>> cacheSpec,
            @ParameterName(name="criteriaSetLookupStrategy") @Nullable 
            final Function<ProfileRequestContext, CriteriaSet> criteriaSetLookupStrategy,
            @ParameterName(name="id") @Nonnull final String id) throws ComponentInitializationException, IOException{
        
        final MetadataCacheBuilder.Builder<String, Map<String, MetadataPolicy>> builder = 
                new MetadataCacheBuilder.Builder<>();
        
        Resource fileResource = null;
        if (resource != null && !resource.isEmpty()) {
            final ResourceLoader resourceLoader = new PreferFileSystemResourceLoader();
            fileResource = resourceLoader.getResource(resource);
        }
        
        final DefaultFileLoadingStrategy fileStrategy = new DefaultFileLoadingStrategy(fileResource);
        cacheSpec.setLoadingStrategy(fileStrategy);
        cacheSpec.setCacheId(id + "-cache");

        final MetadataCache<Map<String, MetadataPolicy>> cache = builder.build(cacheSpec);
        
        final OIDCMetadataPolicyResolver resolver = new OIDCMetadataPolicyResolver(cache);
        resolver.setId(id + "-resolver");
        resolver.initialize();
        
        final ResolverBasedRegistrationMetadataPolicyLookupFunction function = 
                new ResolverBasedRegistrationMetadataPolicyLookupFunction();
        function.setMetadataPolicyResolver(resolver);
        function.setCriteriaSetLookupStrategy(criteriaSetLookupStrategy);

        return function;
        
    }
    

}
