/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.impl.DefaultBatchBackingStore;
import net.shibboleth.oidc.metadata.impl.DefaultDynamicBackingStore;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;

/**
 * A builder that creates, initializes, and safely publishes the correct {@link MetadataCache metadata cache} 
 * based on the supplied {@link MetadataCacheBuilderSpec metadata cache builder specification}.
 */
public final class MetadataCacheBuilder {
    
    /** Private constructor.*/
    private MetadataCacheBuilder() {
        
    }
    
    /**
     * A static builder for generating a batch metadata cache from a given specification.
     *
     * @param <IdentifierType> The identifier type 
     * @param <MetadataType> The metadata type
     */
    public static class Builder<IdentifierType, MetadataType> {
        
        /** Class logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(MetadataCacheBuilder.class);
        
    
        /**
         * Build the metadata cache from the given specification.
         * 
         * @param specification the metadata cache specification.
         * 
         * @return the batch metadata cache.
         * 
         * @throws ComponentInitializationException on error.
         */
        public MetadataCache<MetadataType> build(
                @Nonnull final MetadataCacheBuilderSpec<IdentifierType, MetadataType> specification) 
                        throws ComponentInitializationException {
    
            if (specification instanceof BatchMetadataCacheBuilderSpec) {
                final BatchMetadataCacheBuilderSpec<IdentifierType, MetadataType> spec = 
                        (BatchMetadataCacheBuilderSpec<IdentifierType, MetadataType>) specification;
                final BatchMetadataCache<IdentifierType, MetadataType> cache = 
                        new BatchMetadataCache<>(
                        new DefaultBatchBackingStore<>());            
                cache.setSourceMetadataExpiryStrategy(spec.getSourceMetadataExpiryStrategy());
                cache.setLoadingStrategy(spec.getLoadingStrategy());
                cache.setParsingStrategy(spec.getParsingStrategy());
                cache.setMinRefreshDelay(spec.getMinRefreshDelay());
                cache.setMaxRefreshDelay(spec.getMaxRefreshDelay());
                cache.setRefreshDelayFactor(spec.getRefreshDelayFactor());
                cache.setIdentifierExtractionStrategy(spec.getIdentifierExtractionStrategy());
                cache.setCriteriaToIdentifierStrategy(spec.getCriteriaToIdentifierStrategy());
                cache.setMetadataFilterStrategy(spec.getMetadataFilterStrategy());
                cache.setMetadataBeforeRemovalHook(spec.getMetadataBeforeRemovalHook());
                cache.setMatchOnIdentifierRequired(spec.isMatchOnIdentifierRequired());
                cache.setMetadataValidPredicate(spec.getMetadataValidPredicate());
                cache.setSourceMetadataValidPredicate(spec.getSourceMetadataValidPredicate());
                cache.setId(spec.getCacheId());
                cache.initialize();
                return cache;
            } else if (specification instanceof DynamicMetadataCacheBuilderSpec) {
                final DynamicMetadataCacheBuilderSpec<IdentifierType, MetadataType> spec = 
                        (DynamicMetadataCacheBuilderSpec<IdentifierType, MetadataType>) specification;
                final DynamicMetadataCache<IdentifierType, MetadataType> cache = new DynamicMetadataCache<>(
                        new DefaultDynamicBackingStore<>());
                cache.setFetchStrategy(spec.getFetchStrategy());
                cache.setMinCacheDuration(spec.getMinCacheDuration());
                cache.setMaxCacheDuration(spec.getMaxCacheDuration());
                cache.setRefreshDelayFactor(spec.getRefreshDelayFactor());
                cache.setMaxIdleEntityData(spec.getMaxIdleEntityData());
                cache.setMetadataExpirationTimeStrategy(spec.getMetadataExpirationTimeStrategy());
                cache.setIdentifierExtractionStrategy(spec.getIdentifierExtractionStrategy());
                cache.setCriteriaToIdentifierStrategy(spec.getCriteriaToIdentifierStrategy());
                cache.setCleanupTaskInterval(spec.getCleanupTaskInterval());
                cache.setRemoveIdleEntityData(spec.isRemoveIdleEntityData());
                cache.setInitialCleanupTaskDelay(spec.getInitialCleanupTaskDelay());
                cache.setMetadataFilterStrategy(spec.getMetadataFilterStrategy());
                cache.setMetadataBeforeRemovalHook(spec.getMetadataBeforeRemovalHook());
                cache.setMetadataValidPredicate(spec.getMetadataValidPredicate());
                cache.setId(spec.getCacheId());
                cache.initialize();
                return cache;
            } else if (specification instanceof FetchThroughMetadataCacheBuilderSpec) {
                final FetchThroughMetadataCacheBuilderSpec<IdentifierType, MetadataType> spec = 
                        (FetchThroughMetadataCacheBuilderSpec<IdentifierType, MetadataType>) specification;
                final FetchThroughMetadataCache<IdentifierType, MetadataType> cache = new FetchThroughMetadataCache<>();
                cache.setFetchStrategy(spec.getFetchStrategy());
                //TODO refresh delay is not really needed here.
                cache.setRefreshDelayFactor(spec.getRefreshDelayFactor());         
                cache.setIdentifierExtractionStrategy(spec.getIdentifierExtractionStrategy());
                cache.setCriteriaToIdentifierStrategy(spec.getCriteriaToIdentifierStrategy());           
                cache.setMetadataFilterStrategy(spec.getMetadataFilterStrategy());
                cache.setMetadataBeforeRemovalHook(spec.getMetadataBeforeRemovalHook());
                cache.setMetadataValidPredicate(spec.getMetadataValidPredicate());
                cache.setId(spec.getCacheId());
                cache.initialize();
                return cache;
            }
            log.error("Unable to construct metadata cache, unknown specification type '{}'",
                    specification.getClass().getSimpleName());
            throw new ComponentInitializationException("Cache Specification type not recognized");
        }
    }

}
