/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A specification for building a fetch-through metadata cache.
 *
 * @param <IdentifierType> The metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
public class FetchThroughMetadataCacheBuilderSpec <IdentifierType, MetadataType> 
                        extends BaseMetadataCacheBuilderSpec<IdentifierType, MetadataType>
                            implements MetadataCacheBuilderSpec<IdentifierType, MetadataType> {
    
    
    /** The function to use to fetch metadata if either none exists, or the existing is stale.*/
    @Nullable private Function<CriteriaSet, MetadataType> fetchStrategy;
 
    
    /** Constructor. */
    protected FetchThroughMetadataCacheBuilderSpec() {
        super();
    }  
    
    /**
     * Set the metadata fetching strategy.
     * 
     * @param strategy the strategy to use.
     */
    public void setFetchStrategy(@Nonnull final Function<CriteriaSet, MetadataType> strategy) {
        fetchStrategy = Constraint.isNotNull(strategy, "Metadata fetch strategy can not be null");
    }
    
    /**
     * Get the metadata fetching strategy.
     * 
     * @return the fetching strategy.
     */
    @Nullable protected Function<CriteriaSet, MetadataType> getFetchStrategy() {
        return fetchStrategy;
    }
    

}
