/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.oidc.metadata.cache.ExpirationTimeContext;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A specification for building a dynamic read-through metadata cache.
 *
 * @param <IdentifierType> The metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
public class DynamicMetadataCacheBuilderSpec <IdentifierType, MetadataType> 
                        extends BaseMetadataCacheBuilderSpec<IdentifierType, MetadataType> 
                            implements MetadataCacheBuilderSpec<IdentifierType, MetadataType> {
    
    
    /** The function to use to fetch metadata if either none exists, or the existing is stale.*/
    @Nullable private Function<CriteriaSet, MetadataType> fetchStrategy;
    
    /** Maximum cache duration. */
    @Nonnull private Duration maxCacheDuration;
    
    /** Minimum cache duration. */
    @Nonnull private Duration minCacheDuration;
    
    /** The maximum idle time for which the cache will keep data for before it is removed. */
    @Nullable private Duration maxIdleEntityData;
    
    /** The interval at which the cleanup task should run. */
    @Nonnull private Duration cleanupTaskInterval;
    
    /** Flag indicating whether idle entity data should be removed. */
    private boolean removeIdleEntityData;
    
    /** The initial cleanup task delay.*/
    @Nonnull private Duration initialCleanupTaskDelay;
    
    /** Strategy used to compute an expiration time. */
    @Nullable private Function<ExpirationTimeContext<MetadataType>, Instant> metadataExpirationTimeStrategy;
    
    
    /** Constructor. */
    protected DynamicMetadataCacheBuilderSpec() {
        super();
        maxCacheDuration = Duration.ofHours(8);
        minCacheDuration = Duration.ofMinutes(10); 
        maxIdleEntityData = Duration.ofHours(8);
        cleanupTaskInterval = Duration.ofMinutes(30);
        initialCleanupTaskDelay = Duration.ofMinutes(1);
        removeIdleEntityData = true;
    }
    
    
    /**
     *  Get the maximum cache duration for metadata.
     *  
     *  <p>Defaults to: 8 hours.</p>
     *  
     * @return the maximum cache duration
     */
    @Nonnull protected Duration getMaxCacheDuration() {
        return maxCacheDuration;
    }

    /**
     *  Set the maximum cache duration for metadata.
     *  
     *  <p>Defaults to: 8 hours.</p>
     *  
     * @param duration the maximum cache duration
     */
    public void setMaxCacheDuration(@Nonnull final Duration duration) {        
        Constraint.isNotNull(duration, "Duration cannot be null");
        Constraint.isFalse(duration.isNegative(), "Duration cannot be negative");
        
        maxCacheDuration = duration;
    }
    
    /**
     *  Get the minimum cache duration for metadata.
     *  
     *  <p>Defaults to: 10 minutes.</p>
     *  
     * @return the minimum cache duration
     */
    @Nonnull protected Duration getMinCacheDuration() {
        return minCacheDuration;
    }
    

    /**
     *  Set the minimum cache duration for metadata.
     *  
     *  <p>Defaults to: 10 minutes.</p>
     *  
     * @param duration the minimum cache duration
     */
    public void setMinCacheDuration(@Nonnull final Duration duration) {
        Constraint.isNotNull(duration, "Duration cannot be null");
        Constraint.isFalse(duration.isNegative(), "Duration cannot be negative");
        
        minCacheDuration = duration;
    }
    
    /**
     * Set the flag indicating whether idle entity data should be removed. 
     * 
     * @param flag true if idle entity data should be removed, false otherwise
     */
    public void setRemoveIdleEntityData(final boolean flag) {       
        removeIdleEntityData = flag;
    }
    
    /**
     * Should idle metadata be removed?
     * 
     * @return if idle metadata should be resolved.
     */
    protected boolean isRemoveIdleEntityData() {
        return removeIdleEntityData;
    }
    
    /**
     * Set the metadata expiration time strategy.
     * 
     * @param strategy the strategy.
     */
    public void setMetadataExpirationTimeStrategy(
            @Nonnull final Function<ExpirationTimeContext<MetadataType>, Instant> strategy) {        
        metadataExpirationTimeStrategy = Constraint.isNotNull(strategy, "Strategy can not be null");
    }
    
    /**
     * Get the metadata expiration time strategy.
     * 
     * @return the strategy.
     */
    @Nullable protected Function<ExpirationTimeContext<MetadataType>, Instant> getMetadataExpirationTimeStrategy() {
        return metadataExpirationTimeStrategy;
    }
    
    /**
     * Get the interval at which the cleanup task should run.
     * 
     * <p>Defaults to: 30 minutes.</p>
     * 
     * @return return the interval
     */
    @Nonnull protected Duration getCleanupTaskInterval() {
        return cleanupTaskInterval;
    }

    /**
     * Set the interval at which the cleanup task should run.
     * 
     * <p>Defaults to: 30 minutes.</p>
     * 
     * @param interval the interval to set
     */
    public void setCleanupTaskInterval(@Nonnull final Duration interval) {        
        Constraint.isNotNull(interval, "Cleanup task interval may not be null");
        Constraint.isFalse(interval.isNegative() || interval.isZero(), "Cleanup task interval must be positive");
        
        cleanupTaskInterval = interval;
    }
    
    /**
     * Set the metadata fetching strategy.
     * 
     * @param strategy the strategy to use.
     */
    public void setFetchStrategy(@Nonnull final Function<CriteriaSet, MetadataType> strategy) {
        fetchStrategy = Constraint.isNotNull(strategy, "Metadata fetch strategy can not be null");
    }
    
    /**
     * Get the metadata fetching strategy.
     * 
     * @return the fetching strategy.
     */
    @Nullable protected Function<CriteriaSet, MetadataType> getFetchStrategy() {
        return fetchStrategy;
    }
    
    /**
     * Get the maximum idle time for which the resolver will keep data for before it is removed.
     * 
     * <p>Defaults to: 8 hours.</p>
     * 
     * @return return the maximum idle time
     */
    @Nonnull protected Duration getMaxIdleEntityData() {
        return maxIdleEntityData;
    }

    /**
     * Set the maximum idle time for which the resolver will keep data for before it is removed.
     * 
     * <p>Defaults to: 8 hours.</p>
     * 
     * @param max the maximum entity data idle time
     */
    public void setMaxIdleEntityData(@Nonnull final Duration max) {
        Constraint.isNotNull(max, "Max idle time cannot be null");
        Constraint.isFalse(max.isNegative(), "Max idle time cannot be negative");

        maxIdleEntityData = max;
    }
    
    
    /**
     * Get the initial cleanup task delay.
     * 
     * @return Returns the initialCleanupTaskDelay.
     */
    @Nonnull protected Duration getInitialCleanupTaskDelay() {
        return initialCleanupTaskDelay;
    }

    /**
     * Set the initial cleanup task delay.
     * 
     * @param delay The initialCleanupTaskDelay to set.
     */
    public void setInitialCleanupTaskDelay(@Nonnull final Duration delay) {
        Constraint.isNotNull(delay, "Cleanup task delay can not be null");
        Constraint.isFalse(delay.isNegative() || delay.isZero(), "Cleanup task delay must be positive");
        initialCleanupTaskDelay = delay;
        
    }

}
