/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;

/** Parse a raw byte array into a *single* OIDCProviderMetadata object.*/
public class DefaultOIDCProviderMetadataParsingStrategy implements Function<byte[], List<OIDCProviderMetadata>> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultOIDCProviderMetadataParsingStrategy.class);

    @Override
    @Nonnull public List<OIDCProviderMetadata> apply(@Nullable final byte[] rawMetdata) {
        if (rawMetdata == null) {
            log.warn("Raw metadata is null, unable to parse OIDC Provider Metadata");
            return Collections.emptyList();
        }
        try {
            final OIDCProviderMetadata metadata =
                    OIDCProviderMetadata.parse(new String(rawMetdata,StandardCharsets.UTF_8));
            if (metadata != null) {
                return List.of(metadata);
            }
            return Collections.emptyList();
        } catch (final ParseException e) {
            log.error("Error parsing bytes to metadata", e);
            return Collections.emptyList();
        }
    }

}
