/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

import net.shibboleth.oidc.metadata.cache.CacheLoadingContext;
import net.shibboleth.oidc.metadata.cache.CacheLoadingException;
import net.shibboleth.oidc.metadata.cache.LoadingStrategy;
import net.shibboleth.oidc.metadata.impl.ResolverHelper;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;

/** Default strategy for loading information from a file.*/
@ThreadSafe
public class DefaultFileLoadingStrategy implements LoadingStrategy {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultFileLoadingStrategy.class);
    
    /** The metadata file. */
    @Nullable private final File metadataFile;
    
    /** The metadata file name to use in logs. */
    @Nonnull @NotEmpty private final String metadataFileFriendlyName;
    
    /**
     * 
     * Constructor.
     *
     * @param metadata the metadata file resource. Can be {@literal null}.
     * 
     * @throws IOException if the file is not null put does not exist.
     */
    public DefaultFileLoadingStrategy(@Nullable final Resource metadata) throws IOException {
       if (metadata == null) {
           log.warn("File resource is null, no bytes will be returned");
           metadataFile = null;
           metadataFileFriendlyName = "No file specified";
       } else {
           metadataFile = metadata.getFile();
           metadataFileFriendlyName = metadata.getDescription();
       }
    }
    
    /**
     * Get the time for the last update/modification of the metadata file.
     * @return The last update time.
     */
    private Instant getMetadataUpdateTime() {
        return Instant.ofEpochMilli(metadataFile.lastModified());
    }

    @Override
    @Nullable public byte[] load(@Nonnull final CacheLoadingContext context) throws CacheLoadingException {
        if (metadataFile == null) {
            return null;
        }
        try {
            ResolverHelper.validateMetadataFile(metadataFile);
            final Instant metadataUpdateTime = getMetadataUpdateTime();
            if (context.getLastRefresh() == null || context.getLastUpdate() == null || 
                    metadataUpdateTime.isAfter(context.getLastRefresh())) {
                log.debug("Returning the contents of {} as byte array", metadataFile.toPath());
                return ResolverHelper.inputstreamToByteArray(new FileInputStream(metadataFile));
            }
            return null;
        } catch (final Exception e) {
            final String errMsg = "Unable to read metadata file " + metadataFile.getAbsolutePath();
            log.error(errMsg, e.getMessage());
            throw new CacheLoadingException(errMsg, e);
        }
    }

    @Override
    @Nonnull public String getSourceIdentifier() {
       return metadataFileFriendlyName;
    }

}
