/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.ThreadFactoryBuilder;

import net.shibboleth.oidc.metadata.BackingStore;
import net.shibboleth.oidc.metadata.MetadataManagementData;
import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.cache.MetadataCacheException;
import net.shibboleth.oidc.metadata.filter.MetadataFilterContext;
import net.shibboleth.oidc.metadata.filter.MetadataSource;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.primitive.TimerSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;


/**
 * A base {@link MetadataCache} implementation. Supports the following:
 * <ul>
 * <li>A configurable backing store to store metadata.</li>
 * <li>Individual entity metadata filtering during cache save.</li>
 * <li>Configuration of type specific functions via strategies.</li>
 * </ul>
 *
 * @param <IdentifierType> the metadata identifier type
 * @param <MetadataType> the metadata type
 */
public abstract class AbstractMetadataCache<IdentifierType, MetadataType> 
                extends AbstractIdentifiableInitializableComponent implements MetadataCache<MetadataType> {
    
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(AbstractMetadataCache.class);    
    
    /** Cached log prefix. */
    @Nullable private String logPrefix;
    
    /** Factor used to compute when the next refresh interval will occur. Default value: 0.75 */
    @NonnullAfterInit @Positive private Float refreshDelayFactor;

    /** Backing store for runtime metadata.*/
    @Nullable private final BackingStore<IdentifierType, MetadataType> backingStore; 
    
    /** 
     * A hook that is executed just before a cache entry will been removed/invalidated/evicted.
     */
    @Nullable private BiConsumer<List<MetadataType>, IdentifierType> metadataBeforeRemovalHook;
    
    /** Strategy used to extract an identifier from the given metadata.*/
    @NonnullAfterInit private Function<MetadataType, IdentifierType> identifierExtractionStrategy;
    
    /** Map criteria to identifiers to use as keys to the backing store.*/
    @NonnullAfterInit private Function<CriteriaSet, IdentifierType> criteriaToIdentifierStrategy;
    
    /** A strategy to filter metadata. */
    @NonnullAfterInit private BiFunction<MetadataType, MetadataFilterContext , MetadataType> metadataFilterStrategy;
    
    /** Is the metadata valid? */
    @NonnullAfterInit private Predicate<MetadataType> metadataValidPredicate;
    
    /** A single threaded executor service for running background cache tasks.*/
    @NonnullAfterInit private ScheduledExecutorService executorService;  
    
    
    /** Whether we created our own schedular during object construction. */
    private boolean createOwnSchedular;
    
    /** 
     * Package private constructor. For safe-construction through the factory only.
     * 
     * @param store the metadata backing store.
     */
    AbstractMetadataCache(@Nullable final BackingStore<IdentifierType, MetadataType> store) {       
        this(store, null);
    }
    
   
    /**
     * 
     * Package private constructor. For safe-construction through the factory only.
     * 
     * <p>Accepts an executor. Mostly used for testing.</p>
     *
     * @param store the backing store.
     * @param executor the scheduled executor
     */
    AbstractMetadataCache(@Nullable final BackingStore<IdentifierType, MetadataType> store, 
            @Nullable final ScheduledExecutorService executor) {    
        backingStore = store;
        if (executor != null) {
            executorService = executor;
            createOwnSchedular = false;
        } else {
            createOwnSchedular = true;
        }
        
    }
    
    /**
     * Return a prefix for logging messages for this component.
     * 
     * @return a string for insertion at the beginning of any log messages
     */
    @Nonnull @NotEmpty protected String getLogPrefix() {
        if (logPrefix == null) {
            logPrefix = "Metadata Cache " + (getId() != null ? getId() : "(unknown)") + ":";
        }
        return logPrefix;
    }

    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();    
       
        
        if (identifierExtractionStrategy == null) {
            throw new ComponentInitializationException("Identifier extraction strategy can not be null");
        }

        if (criteriaToIdentifierStrategy == null) {
            throw new ComponentInitializationException("Criteria to identifier strategy can not be null");
        }
        if (metadataFilterStrategy == null) {
            throw new ComponentInitializationException("Metadata filter strategy can not be null");
        }
        if (refreshDelayFactor == null) {
            throw new ComponentInitializationException("Metadata cache not property initialized");
        }
        if (metadataValidPredicate == null) {
            throw new ComponentInitializationException("Metadata validation predicate can not be null");
        }
        
        // create a schedular for thread tasks.
        if (createOwnSchedular) {
            // Use thread builder to allow setting threads as deamon and a name.
            // Set as deamon background threads. Do not prevent JVM exit.
            executorService = Executors.newSingleThreadScheduledExecutor(
                    new ThreadFactoryBuilder().setDaemon(true).setNameFormat(TimerSupport.getTimerName(this)+"-%d")
                    .build()); 
        }                  
    }
    
    @Override protected void doDestroy() {
        executorService.shutdown();
        super.doDestroy();
    }
    
    /**
     * Get the background executor service.
     * 
     * @return the executor service.
     */
    @NonnullAfterInit protected ScheduledExecutorService getExecutorService() {
        return executorService;
    }
    
    /**
    * Set the predicate which determines if a piece of metadata is valid or not.
    * 
    * @param predicate the predicate.
    */
   public void setMetadataValidPredicate(@Nonnull final Predicate<MetadataType> predicate) {
       ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
       ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
       
      metadataValidPredicate = Constraint.isNotNull(predicate, "Is metadata valid predicate can not be null");
   }
   
   /**
    * Get the predicate which determines if a piece of metadata is valid or not.
    * 
    * @return the predicate.
    */
   @Nonnull protected Predicate<MetadataType> getMetadataValidPredicate() {
       return metadataValidPredicate;
   }

    /**
     * Set the {@link CriteriaSet} to IdentifierType strategy.
     * 
     * @param strategy the strategy.
     */
    public void setCriteriaToIdentifierStrategy(@Nonnull final Function<CriteriaSet, IdentifierType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        criteriaToIdentifierStrategy =  
                Constraint.isNotNull(strategy,"Criteria to identifier strategy can not be null");
    }
    
    /**
     * Get the identifier from criteria set extraction strategy.
     * 
     * @return the identifier extraction strategy. 
     */
    @NonnullAfterInit protected Function<CriteriaSet, IdentifierType> getCriteriaToIdentifierStrategy() {
        return criteriaToIdentifierStrategy;
    }
    
    /**
     * Set the MetadataType to IdentifierType extraction time strategy.
     * 
     * @param strategy the strategy.
     */
    public void setIdentifierExtractionStrategy(@Nonnull final Function<MetadataType, IdentifierType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        identifierExtractionStrategy = Constraint.isNotNull(strategy, "Identifier extraction strategy can not be null");
    }
    
    /**
     * Get the identifier extraction strategy.
     * 
     * @return the identifier extraction strategy.
     */
    @NonnullAfterInit protected Function<MetadataType, IdentifierType> getIdentifierExtractionStrategy() {
        return identifierExtractionStrategy;
    }
    
    /**
     * Set the metadata filtering strategy.
     * 
     * @param strategy the metadata filtering strategy.
     */
    public void setMetadataFilterStrategy(
            @Nonnull final BiFunction<MetadataType, MetadataFilterContext, MetadataType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        metadataFilterStrategy = Constraint.isNotNull(strategy, "Metadata filter strategy can not be null");
    }
    
    /**
     * Get the metadata filtering strategy.
     * 
     * @return the filtering strategy.
     */
    @NonnullAfterInit 
    protected BiFunction<MetadataType, MetadataFilterContext, MetadataType> getMetadataFilterStrategy() {
        return metadataFilterStrategy;
    }
    
    /**
     * Set a hook to run before a metadata cache entry is removed from the cache.
     * <p>The hook is required to gracefully handle null metadata lists.</p>
     * 
     * @param hook the hook to run.
     */
    public void setMetadataBeforeRemovalHook(
            @Nullable final BiConsumer<List<MetadataType>, IdentifierType> hook) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        metadataBeforeRemovalHook = hook;
    }
    

    /**
     * Sets the delay factor used to compute the next refresh time. The delay must be between 0.0 and 1.0, exclusive.
     * 
     * <p>Defaults to:  0.75.</p>
     * 
     * @param factor delay factor used to compute the next refresh time
     */
    public void setRefreshDelayFactor(@Nonnull final Float factor) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        if (factor <= 0 || factor >= 1) {
            throw new 
                ConstraintViolationException("Refresh delay factor must be a number between 0.0 and 1.0, exclusive");
        }

        refreshDelayFactor = factor;
    }
    
    /**
     * Get the refresh delay factor.
     * 
     * @return the refresh delay factor.
     */
    @NonnullAfterInit protected Float getRefreshDelayFactor() {
        return refreshDelayFactor;
    }
    
    /**
     * Get list of descriptors matching the identifier.
     * 
     * @param identifier the identifier to lookup
     * 
     * @return a list of metadata
     * 
     * @throws MetadataCacheException if an error occurs.
     */
    @Nonnull @NonnullElements protected List<MetadataType> lookupIdentifier(
            @Nonnull @NotEmpty final IdentifierType identifier) throws MetadataCacheException {
        
        if (!isInitialized()) {
            throw new MetadataCacheException("Metadata resolver has not been initialized");
        }
        
        final List<MetadataType> metadata = lookupIndexedIdentifier(identifier);
       
        if (metadata.isEmpty()) {
            log.debug("{} Metadata cache does not contain an entry for '{}'", 
                    getLogPrefix(), identifier);
            return metadata;
        }
        final Iterator<MetadataType> metadataIter = metadata.iterator();
        
        while (metadataIter.hasNext()) {
            final MetadataType individualMetadata = metadataIter.next();
            if (!metadataValidPredicate.test(individualMetadata)) {
                log.warn("{} Metadata cache contained an entry with the identifier '{}', " 
                        + " but it was no longer valid", getLogPrefix(), identifier);
                metadataIter.remove();
            }
        }
        if (!metadata.isEmpty()) {
            log.trace("{} Metadata cache has found '{}' entry(s) for '{}'", 
                    getLogPrefix(), metadata.size(), identifier);
        }
        return metadata;
    }
    
    /**
     * Lookup the specified entityID from the index. The returned list will be a copy of what is stored in the backing
     * index, and is safe to be manipulated by callers.
     * 
     * @param identifier the identifier to lookup
     * 
     * @return list copy of indexed metadata, may be empty, will never be null
     */
    @Nonnull @NonnullElements protected List<MetadataType> lookupIndexedIdentifier(
            @Nonnull final IdentifierType identifier) {
        final List<MetadataType> metadata = getBackingStore().getIndexedValues().get(identifier);
        if (metadata != null) {
            return new ArrayList<>(metadata);
        }
        return Collections.emptyList();
    }
    
    /**
     * Get the backing store.
     * 
     * @return the backing store. Can be {@literal null}.
     */
    @Nullable protected BackingStore<IdentifierType, MetadataType> getBackingStore() {
        return backingStore;
    }
    
     
    /**
     * Clear the backing store and load each metadata entry one at a time.
     * 
     * @param metadataToStore the metadata to load into the cache.
     */
    protected void freshLoad(@Nonnull final List<MetadataType> metadataToStore) {
        invalidateAll();
        for (final MetadataType metadata : metadataToStore) {           
            writeToBackingStore(metadata);
        }        
    }
    
    /**
     * Write the given metadata to the backing store index.
     * 
     * <p>Should happen within an appropriate lock to ensure thread-safety.</p>
     * 
     * @param metadata the metadata to add to the backing store.
     */
    protected void writeToBackingStore(@Nonnull final MetadataType metadata) {
        
        final IdentifierType extractedIdentifier = getIdentifierExtractionStrategy().apply(metadata);
        
        if (extractedIdentifier == null) {
            log.warn("{} Identifier could not be extracted from metadata, metadata not stored",getLogPrefix());
            return;
        }
        
        // will only effect the dynamic cache case, static loading should have cleared the backing store
        // by this point
        invalidate(extractedIdentifier);
        
        backingStore.getOrderedValues().add(metadata);
        // add new metadata to index
        List<MetadataType> existingMetadata = backingStore.getIndexedValues().get(extractedIdentifier);
        if (existingMetadata == null) {
            existingMetadata = new ArrayList<>();
            backingStore.getIndexedValues().put(extractedIdentifier, existingMetadata);
        } else if (!existingMetadata.isEmpty()) {
            log.warn("{} Detected duplicate metadata for identifier: {}", getLogPrefix(), extractedIdentifier);
        }
        existingMetadata.add(metadata);
    }    
    
    
    /**
     * Get a new instance of {@link MetadataFilterContext} to be used when filtering metadata.
     *
     * <p>
     * This default implementation just returns an empty context with as metadatasource for passing to 
     * the filter strategy. The strategy can then add to the context as appropriate.
     * </p>
     *
     * @return the new filter context instance
     */
    @Nonnull protected MetadataFilterContext newFilterContext() {
        
        final MetadataSource source = new MetadataSource();
        source.setSourceId(getId());

        final MetadataFilterContext context = new MetadataFilterContext();
        context.add(source);
        
        return context;
    }
    
   
    /**
     * Determine if the metadata should be refreshed based on stored refresh trigger time. 
     * 
     * @param mgmtData the entity'd management data
     * @return true if should attempt refresh, false otherwise
     */
    protected boolean shouldAttemptRefresh(@Nonnull final MetadataManagementData<IdentifierType> mgmtData) {
        return Instant.now().isAfter(mgmtData.getRefreshTriggerTime());
        
    }
    
    /**
     * Determine if the metadata has expired based on the expiration time set in the managment metadata.
     *
     * @param mgmtData management data
     * 
     * @return true iff the metadata has expired
     */
    protected boolean hasExpired(@Nonnull final MetadataManagementData<IdentifierType> mgmtData) {
        return Instant.now().isAfter(mgmtData.getExpirationTime());
    }
    
    
    /**
     * Remove/discard from the backing store all metadata for the entity with the given identifier.
     * 
     * @param identifier the ID of the metadata to remove
     */
    protected void invalidate(@Nonnull final IdentifierType identifier) {
        final Map<IdentifierType, List<MetadataType>> indexedDescriptors = backingStore.getIndexedValues();
        final List<MetadataType> descriptors = indexedDescriptors.get(identifier);
        
        if (descriptors != null) {            
            if (metadataBeforeRemovalHook != null) {
                metadataBeforeRemovalHook.accept(descriptors, identifier);
            }            
            backingStore.getOrderedValues().removeAll(descriptors);
            indexedDescriptors.remove(identifier);
        }
        
    }
    
    /**
     * Remove/discard all metadata for the backing store.
     * 
     * <p>Ensure thread-safety is observed in the calling method.</p>
     * 
     * TODO check lock. 
     */
    protected void invalidateAll() {
        backingStore.getIndexedValues().clear();
        backingStore.getOrderedValues().clear();
    }
    
    
    /**
     * Create a wrapper for runnables that catches any throwable and logs it. Useful
     * to prevent thread death from an uncaught exception.
     * 
     * @param action the runnable to wrap
     * 
     * @return the wrapped runnable.
     */
    @Nonnull protected Runnable errorHandlingWrapper(@Nonnull final Runnable action) {
        return () -> {
            try {
                action.run();
            } catch (final Throwable e) {
                log.error("{} Error executing thread task", getLogPrefix(), e);
            }
        };
    }
    
    
}

