/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.MoreObjects;

/**
 * The metadata policy for a single claim. It exploits the operators defined in the 5.1 of OIDC federation spec
 * (draft 17 for time being): https://openid.net/specs/openid-connect-federation-1_0.html#rfc.section.5.1
 */
public class MetadataPolicy {
    
    /** The (forced) value for the claim. */
    @JsonProperty("value") private Object forcedValue;
    
    /** The value(s) to be added for the claim. */
    @JsonProperty("add") private Object add;
    
    /** The default value for the claim to be used in case none is specified. */
    @JsonProperty("default") private Object defaultValue;
    
    /** The list of values from which the claim must be one of. */
    @JsonProperty("one_of") private List<Object> oneOfValues;

    /** The list of value(s) from which the claim must be subset of. */
    @JsonProperty("subset_of") private List<Object> subsetOfValues;
    
    /** The list of value(s) from which the claim must be superset of. */
    @JsonProperty("superset_of") private List<Object> supersetOfValues;    
    
    /** The flag indicating that the claim must have a value. */
    @JsonProperty("essential") private Boolean essential;
    
    /** The regular expression that the claim value must meet. */
    @JsonProperty("regexp") private String regexp;

    /**
     * Get the (forced) value for the claim.
     * 
     * @return The (forced) value for the claim.
     */
    public Object getValue() {
        return forcedValue;
    }

    /**
     * Set the (forced) value for the claim.
     * 
     * @param value What to set.
     */
    public void setValue(final Object value) {
        this.forcedValue = value;
    }

    /**
     * Get the value(s) to be added for the claim.
     * 
     * @return The value(s) to be added for the claim.
     */
    public Object getAdd() {
        return add;
    }

    /**
     * Set the value(s) to be added for the claim.
     * 
     * @param value What to set.
     */
    public void setAdd(final Object value) {
        this.add = value;
    }

    /**
     * Get the default value for the claim to be used in case none is specified.
     * 
     * @return The default value for the claim to be used in case none is specified.
     */
    public Object getDefaultValue() {
        return defaultValue;
    }

    /**
     * Set the default value for the claim to be used in case none is specified.
     * 
     * @param value What to set.
     */
    public void setDefaultValue(final Object value) {
        this.defaultValue = value;
    }

    /**
     * Get the list of values from which the claim must be one of.
     * 
     * @return The list of values from which the claim must be one of.
     */
    public List<Object> getOneOfValues() {
        return oneOfValues;
    }

    /**
     * Set the list of values from which the claim must be one of.
     * 
     * @param values What to set.
     */
    public void setOneOfValues(final List<Object> values) {
        this.oneOfValues = values;
    }

    /**
     * Get the list of value(s) from which the claim must be subset of.
     * 
     * @return The list of value(s) from which the claim must be subset of.
     */
    public List<Object> getSubsetOfValues() {
        return subsetOfValues;
    }

    /**
     * Set the list of value(s) from which the claim must be subset of.
     * 
     * @param values What to set.
     */
    public void setSubsetOfValues(final List<Object> values) {
        this.subsetOfValues = values;
    }

    /**
     * Get the list of value(s) from which the claim must be superset of.
     * 
     * @return The list of value(s) from which the claim must be superset of.
     */
    public List<Object> getSupersetOfValues() {
        return supersetOfValues;
    }

    /**
     * Set the list of value(s) from which the claim must be superset of.
     * 
     * @param values What to set.
     */
    public void setSupersetOfValues(final List<Object> values) {
        this.supersetOfValues = values;
    }

    /**
     * Get the flag indicating that the claim must have a value.
     * 
     * @return The flag indicating that the claim must have a value.
     */
    public Boolean getEssential() {
        return essential;
    }
    
    /**
     * Is the flag indicating that the claim must have a value enabled.
     * 
     * @return true if enabled, false otherwise.
     */
    public boolean isEssential() {
        return essential == null ? false : essential.booleanValue();
    }

    /**
     * Set the flag indicating that the claim must have a value.
     * 
     * @param isEssential What to set.
     */
    public void setEssential(final Boolean isEssential) {
        this.essential = isEssential;
    }

    /**
     * Get the regular expression that the claim value must meet.
     * 
     * @return The regular expression that the claim value must meet.
     */
    public String getRegexp() {
        return regexp;
    }

    /**
     * Set the regular expression that the claim value must meet.
     * 
     * @param value What to set.
     */
    public void setRegexp(final String value) {
        regexp = value;
    }
    
    /**
     * {@inheritDoc}
     */
    @Override public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("value", forcedValue)
                .add("default", defaultValue)
                .add("add", add)
                .add("one_of", oneOfValues)
                .add("subset_of", subsetOfValues)
                .add("superset_of", supersetOfValues)
                .add("essential", essential)
                .add("regexp", regexp).toString();
    }

    /**
     * Builder class for the {@link MetadataPolicy} objects.
     */
    public static class Builder {
        
        /** The (forced) value for the claim. */
        private Object forcedValue;
        
        /** The value(s) to be added for the claim. */
        private Object add;
        
        /** The default value for the claim to be used in case none is specified. */
        private Object defaultValue;
        
        /** The list of values from which the claim must be one of. */
        private List<Object> oneOfValues;

        /** The list of value(s) from which the claim must be subset of. */
        private List<Object> subsetOfValues;
        
        /** The list of value(s) from which the claim must be superset of. */
        private List<Object> supersetOfValues;    
        
        /** The flag indicating that the claim must have a value. */
        private Boolean essential;
       
        /** The regular expression that the claim value must meet. */
        private String regexp;
        
        /**
         * Constructor.
         */
        public Builder() {
            
        }

        /**
         * Set the (forced) value for the claim.
         * 
         * @param value What to set.
         * @return The builder object with the new value set.
         */
        public Builder withValue(final Object value) {
            this.forcedValue = value;
            return this;
        }
        
        /**
         * Set the value(s) to be added for the claim.
         * 
         * @param value What to set.
         * @return The builder object with the new value set.
         */
        public Builder withAdd(final Object value) {
            this.add = value;
            return this;
        }

        /**
         * Set the default value for the claim to be used in case none is specified.
         * 
         * @param value What to set.
         * @return The builder object with the new value set.
         */
        public Builder withDefaultValue(final Object value) {
            this.defaultValue = value;
            return this;
        }

        /**
         * Set the list of values from which the claim must be one of.
         * 
         * @param values What to set.
         * @return The builder object with the new value set.
         */
        public Builder withOneOfValues(final List<Object> values) {
            this.oneOfValues = values;
            return this;
        }

        /**
         * Set the list of value(s) from which the claim must be subset of.
         * 
         * @param values What to set.
         * @return The builder object with the new value set.
         */
        public Builder withSubsetOfValues(final List<Object> values) {
            this.subsetOfValues = values;
            return this;
        }

        /**
         * Set the list of value(s) from which the claim must be superset of.
         * 
         * @param values What to set.
         * @return The builder object with the new value set.
         */
        public Builder withSupersetOfValues(final List<Object> values) {
            this.supersetOfValues = values;
            return this;
        }

        /**
         * Set the flag indicating that the claim must have a value.
         * 
         * @param isEssential What to set.
         * @return The builder object with the new value set.
         */
        public Builder withEssential(final Boolean isEssential) {
            this.essential = isEssential;
            return this;
        }

        /**
         * Set the regular expression that the claim value must meet.
         * 
         * @param value What to set.
         * @return The builder object with the new value set.
         */
        public Builder withRegexp(final String value) {
            this.regexp = value;
            return this;
        }

        /**
         * Build the metadata policy corresponding to the current builder state.
         * 
         * @return The metadata policy corresponding to the current builder state.
         */
        public MetadataPolicy build() {
            final MetadataPolicy policy = new MetadataPolicy();
            policy.setValue(this.forcedValue);
            policy.setAdd(this.add);
            policy.setDefaultValue(this.defaultValue);
            policy.setOneOfValues(this.oneOfValues);
            policy.setSubsetOfValues(this.subsetOfValues);
            policy.setSupersetOfValues(this.supersetOfValues);
            policy.setEssential(this.essential);
            policy.setRegexp(this.regexp);
            return policy;
        }
    }
    
}
