/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.criterion;

import java.util.Objects;

import javax.annotation.Nonnull;

import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.Criterion;

/**
 * A {@link Criterion} representing a resource location.
 */
public class ResourceLocationCriterion implements Criterion {

    /** The resource location. */
    @Nonnull private final String location;
    
    /**
     * Constructor.
     *
     * @param resourceLocation The resource location.
     */
    public ResourceLocationCriterion(@Nonnull final String resourceLocation) {
        location = Constraint.isNotNull(resourceLocation, "Resource location cannot be null");
    }
    
    /**
     * Get the resource location.

     * @return The resource location.
     */
    @Nonnull public String getResourceLocation() {
        return location;
    }
    
    /** {@inheritDoc} */
    @Override
    public String toString() {
        return "ResourceLocationCriterion [location=" + location + "]";
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode() {
        return Objects.hash(location);
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ResourceLocationCriterion other = (ResourceLocationCriterion) obj;
        return location.equals(other.location);
    }

}
