/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache;

import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Context to hold information pertaining to the computation of a metadata's expiration time.
 * 
 * @param <T> The metadata type. 
 */
@Immutable
public class ExpirationTimeContext<T> {
    
    /** The metadata to base the expiry time off.*/
    @Nonnull private final T metadata;

    /** Minimum cache duration. */
    @Nonnull private final Duration minCacheDuration;
    
    /** Maximum cache duration. */
    @Nonnull private final Duration maxCacheDuration;
    
    /** The now time to base computation off.*/
    @Nonnull private final Instant now;

    /**
     * Constructor.
     *
     * @param metadataIn the metadata to base the expiry time off.
     * @param minimumCacheDuration minimum cache duration.
     * @param maximumCacheDuration maximum cache duration.     * 
     * @param timeNow the now time to base computation off.
     */
    public ExpirationTimeContext(@Nonnull final T metadataIn, 
            @Nonnull final Duration minimumCacheDuration, 
            @Nonnull final Duration maximumCacheDuration, 
            @Nonnull final Instant timeNow) {
        super();
        metadata = Constraint.isNotNull(metadataIn, "Metadata to base expiry off can not be null");
        minCacheDuration =  Constraint.isNotNull(minimumCacheDuration,"Min cache duration can not be null");
        maxCacheDuration =  Constraint.isNotNull(maximumCacheDuration,"Max cache duration can not be null");
        now = timeNow;
    }
    
    /**
     * Get the metadata. 
     * 
     * @return Returns the metadata.
     */
    public final T getMetadata() {
        return metadata;
    }

    /**
     * Get the minimum cache duration.
     * 
     * @return Returns the minCacheDuration.
     */
    public final Duration getMinCacheDuration() {
        return minCacheDuration;
    }

    /**
     * 
     * Get the maximum cache duration. 
     * 
     * @return Returns the maxCacheDuration.
     */
    public final Duration getMaxCacheDuration() {
        return maxCacheDuration;
    }

    /**
     * Get the time to use a 'now'. 
     * 
     * @return Returns now.
     */
    public final Instant getNow() {
        return now;
    }

}
