/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;

/** A dynamic version of a BackingStore.
 * 
 * @param <I> the identifier type
 * @param <T> the type of object stored, referenced by the key.
 */
public interface DynamicBackingStore<I, T> extends BackingStore<I, T> {
    
    /**
     * Get the management data for the specified identifier. If the management data does not exist
     * it should be created using the supplied mapping function.
     * 
     * <p>Management data facilitates per-entity metadata locking and cache primitives e.g. next refresh time. </p>
     * 
     * <p>Should do so in a thread-safe way e.g. if two threads enter this method, only one should be allowed
     * to create management data for the same identifier.</p>
     * 
     * @param identifier the identifier of the entity to find management data about
     * @param mappingFunction the function used to create a new {@link MetadataManagementData} instance if none exist.
     * 
     * @return a new or previously cache metadata management data.
     */
    @Nonnull public MetadataManagementData<I> computeManagementDataIfAbsent(@Nonnull final I identifier,
            @Nonnull final Function<I, MetadataManagementData<I>> mappingFunction);
    
    /**
     * Get the management data for the specified identifier. If the management data does not exist, {@literal null}
     * is returned. 
     * 
     * @param identifier the identifier of the entity to find management data about
     * 
     * @return the corresponding management data, or {@literal null} if not found.
     */
    @Nullable public MetadataManagementData<I> getManagementData(@Nonnull final I identifier);
    
    /**
     * Remove the management data for the specified entityID.
     * 
     * @param identifier the input identifier
     */
    void removeManagementData(@Nonnull final I identifier);
    
    /**
     * Get the set of entityIDs which currently have management data.
     *
     * @return set of entityIDs, may be empty
     */
    @Nonnull @NonnullElements @Unmodifiable @NotLive
    Set<I> getManagementDataIdentifiers();

}
