/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata;

import java.time.Instant;

import javax.annotation.Nullable;

/**
 * A specialisation of a {@link BackingStore} that deals with batch metadata. Operations on batch metadata
 * happen together e.g. a reload will evict all previous entries and all new entries are reloaded in one go. 
 * Individual entries are not updated dynamically as an when needed. 
 *
 * @param <I> The metadata identifier type.
 * @param <T> The metadata type.
 */
public interface BatchBackingStore<I,T> extends BackingStore<I,T> {
    
    
    /**
     * Get the time at which this batch backing store was successfully reloaded. That is, completely cleared and
     * new entries added.
     * 
     * @return the last update time.
     */
    @Nullable Instant getLastUpdate();
    
    
    /**
     * Get the time the last refresh of this backing store was attempted.
     * 
     * @return the last attempted refresh time.
     */
    @Nullable Instant getLastRefresh();
    
    
    /**
     * Set the time at which this batch backing store was successfully reloaded/updated.
     * 
     * @param updatedAt the time it was last updated.
     */
    void setLastUpdate(@Nullable final Instant updatedAt);
    
    /**
     * Set the time at which the last refresh was attempted.
     * 
     * @param refreshedAt the refreshed time. 
     */
    void setLastRefresh(@Nullable final Instant refreshedAt);


    /**
     * Gets the source bytes of the source before it was parsed.
     *  
     * @return the source bytes.
     */
    @Nullable byte[] getOriginalValue();
    
    /**
     * Sets the source bytes of the source before it was parsed.
     *  
     * @param originalValue the source bytes.
     */
    void setOriginalValue(@Nullable final byte[] originalValue);
    
    

}
