/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.function.Predicate;

import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;

/**
 * Activation condition returning true if subject cannot be located from oidc response context. This may be applied
 * preventing the subject resolving in token and userinfo endpoints as it is unnecessary. The activation condition may
 * be used with other protocols also, returning always true.
 */
public class SubjectActivationCondition implements Predicate<ProfileRequestContext> {

    /** {@inheritDoc} */
    @Override
    public boolean test(@Nullable final ProfileRequestContext input) {
        final MessageContext outboundMessageCtx = input.getOutboundMessageContext();
        if (outboundMessageCtx == null) {
            return true;
        }
        final OIDCAuthenticationResponseContext oidcResponseContext =
                outboundMessageCtx.getSubcontext(OIDCAuthenticationResponseContext.class, false);
        if (oidcResponseContext == null) {
            return true;
        }
        return oidcResponseContext.getSubject() == null;
    }

}