/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestCodeChallengeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestCodeChallengeMethodLookupFunction;
import net.shibboleth.oidc.profile.config.logic.AllowPKCEPlainPredicate;
import net.shibboleth.oidc.profile.config.logic.ForcePKCEPredicate;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Validates the presence of PKCE code challenge parameter from the incoming authentication request.
 */
public class ValidateCodeChallenge  extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateCodeChallenge.class);

    /** Strategy used to determine whether to require PKCE. */
    @Nonnull private Predicate<ProfileRequestContext> forcePKCECondition;

    /** Strategy used to determine whether to allow plaintext PKCE. */
    @Nonnull private Predicate<ProfileRequestContext> allowPKCEPlainCondition;
    
    /** Strategy used to locate the code challenge. */
    @Nonnull private Function<ProfileRequestContext, String> codeChallengeLookupStrategy;
    
    /** Strategy used to locate the code challenge method. */
    @Nonnull private Function<ProfileRequestContext, String> codeChallengeMethodLookupStrategy;

    /** Whether PKCE is mandatory. */
    private boolean forcePKCE;

    /** Whether plain PKCE is allowed. */
    private boolean plainPKCE;

    /** PKCE code challenge. */
    @Nullable private String codeChallenge;

    /** PKCE code challenge method. */
    @Nullable private String codeChallengeMethod;

    /**
     * Constructor.
     */
    public ValidateCodeChallenge() {
        forcePKCECondition = new ForcePKCEPredicate();
        allowPKCEPlainCondition = new AllowPKCEPlainPredicate();
        codeChallengeLookupStrategy = new DefaultRequestCodeChallengeLookupFunction();
        codeChallengeMethodLookupStrategy = new DefaultRequestCodeChallengeMethodLookupFunction();
    }

    /**
     * Set the condition used to determine whether to require PKCE.
     * 
     * @param condition condition to apply
     */
    public void setForcePKCECondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        forcePKCECondition = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set the condition used to determine whether to allow plaintext PKCE.
     * 
     * @param condition condition to apply
     */
    public void setAllowPKCEPlainCondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        allowPKCEPlainCondition = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set the strategy used to locate the Code Challenge of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setCodeChallengeLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        codeChallengeLookupStrategy =
                Constraint.isNotNull(strategy, "Code challenge lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the Code Challenge Method of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setCodeChallengeMethodLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        codeChallengeMethodLookupStrategy =
                Constraint.isNotNull(strategy, "Code challenge method lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        forcePKCE = forcePKCECondition.test(profileRequestContext);
        plainPKCE = allowPKCEPlainCondition.test(profileRequestContext);

        codeChallenge = codeChallengeLookupStrategy.apply(profileRequestContext);

        if ((codeChallenge == null || codeChallenge.isEmpty()) && !forcePKCE) {
            log.debug("{} No PKCE code challenge in request, nothing to do", getLogPrefix());
            return false;
        }
        
        codeChallengeMethod = codeChallengeMethodLookupStrategy.apply(profileRequestContext);
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (codeChallenge == null || codeChallenge.isEmpty()) {
            log.warn("{} No PKCE code challenge presented in authentication request" +
                            " even though required by the profile configuration", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.MISSING_PKCE_CODE_CHALLENGE);
            return;
        }
        if (codeChallengeMethod == null || codeChallengeMethod.isEmpty() || "plain".equals(codeChallengeMethod)) {
            if (!plainPKCE) {
                log.warn("{} Plain PKCE code challenge method not allowed", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_PKCE_TRANSFORMATION_METHOD);
                return;
            }
            log.debug("{} Plain code challenge exists in the request and is accepted", getLogPrefix());
        } else if ("S256".equals(codeChallengeMethod)) {
            log.debug("{} S256 code challenge exists in the request", getLogPrefix());
        } else {
            log.warn("{} Unknown code challenge method: {}", getLogPrefix(), codeChallengeMethod);
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_PKCE_TRANSFORMATION_METHOD);
            return;
        }        
    }
}