/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.OutboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCClientRegistrationTokenClaimsContextLookupFunction;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.config.ProfileConfiguration;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;

/**
 * Creates the client ID for the registration.
 * 
 * <p>If an access token was used, it may contain the client_id to use. Otherwise uses
 * the {@link IdentifierGenerationStrategy} attached to the profile configuration. The
 * client ID is included to the {@link OIDCClientRegistrationResponseContext}.</p>
 */
public class GenerateClientID extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(GenerateClientID.class);
    
    /**
     * Strategy used to locate the {@link RelyingPartyContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext,RelyingPartyContext> relyingPartyContextLookupStrategy;

    /** Strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given request. */
    @Nonnull private Function<ProfileRequestContext,OIDCClientRegistrationResponseContext>
        oidcResponseContextLookupStrategy;

    /** Strategy used to locate the {@link OIDCClientRegistrationTokenClaimsContext} associated with the request. */
    @Nonnull private Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext>
        registrationTokenContextLookupStrategy;
    
    /** The RelyingPartyContext to operate on. */
    @Nullable private RelyingPartyContext rpCtx;

    /** The OIDCClientRegistrationResponseContext to create the client ID to. */
    @Nullable private OIDCClientRegistrationResponseContext oidcResponseCtx;

    /** The OIDCClientRegistrationTokenClaimsContext from which to optionally obtain client ID. */
    @Nullable private OIDCClientRegistrationTokenClaimsContext registrationTokenCtx;
    
    /** Constructor. */
    public GenerateClientID() {
        relyingPartyContextLookupStrategy = new ChildContextLookup<>(RelyingPartyContext.class);
        oidcResponseContextLookupStrategy = new ChildContextLookup<>(
                OIDCClientRegistrationResponseContext.class).compose(
                        new OutboundMessageContextLookup());
        registrationTokenContextLookupStrategy = new DefaultOIDCClientRegistrationTokenClaimsContextLookupFunction();
    }

    /**
     * Set the strategy used to locate the {@link RelyingPartyContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link RelyingPartyContext} associated with a given
     *         {@link ProfileRequestContext}
     */
    public void setRelyingPartyContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,RelyingPartyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        relyingPartyContextLookupStrategy = Constraint.isNotNull(strategy,
                "RelyingPartyContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given
     * {@link MessageContext}.
     * 
     * @param strategy strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a 
     *         given {@link MessageContext}
     */
    public void setOidcResponseContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCClientRegistrationResponseContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        oidcResponseContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationResponseContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCClientRegistrationTokenClaimsContext} associated with a given
     * request.
     * 
     * @param strategy lookup strategy
     */
    public void setRegistrationTokenContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        registrationTokenContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationTokenClaimsContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        rpCtx = relyingPartyContextLookupStrategy.apply(profileRequestContext);
        if (rpCtx == null) {
            log.debug("{} No relying party context associated with this profile request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_RELYING_PARTY_CTX);
            return false;
        }

        if (rpCtx.getProfileConfig() == null) {
            log.debug("{} No profile configuration associated with this profile request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_RELYING_PARTY_CTX);
            return false;
        }
        
        if (profileRequestContext.getOutboundMessageContext() == null) {
            log.debug("{} No outbound message context associated with this profile request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;            
        }

        oidcResponseCtx = oidcResponseContextLookupStrategy.apply(profileRequestContext);
        if (oidcResponseCtx == null) {
            log.debug("{} No OIDC client registration response context associated with this profile request", 
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;            
        }
        
        registrationTokenCtx = registrationTokenContextLookupStrategy.apply(profileRequestContext);
        if (registrationTokenCtx != null && registrationTokenCtx.getClaimsSet() == null) {
            registrationTokenCtx = null;
        }

        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final ProfileConfiguration profileConfig = rpCtx.getProfileConfig();
        final IdentifierGenerationStrategy idGenerator =
                profileConfig.getSecurityConfiguration(profileRequestContext).getIdGenerator();
        
        String clientId = null;
        if (registrationTokenCtx != null) {
            clientId = registrationTokenCtx.getClaimsSet().getClientId();
        }
        
        if (clientId != null) {
            log.debug("{} Using client_id supplied by access token: {}", getLogPrefix(), clientId);
        } else {
            clientId = idGenerator.generateIdentifier();
            log.debug("{} Created a new client ID: {}", getLogPrefix(), clientId);
        }
        
        oidcResponseCtx.setClientId(clientId);
    }

}