/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Action that adds authorization code hash claim to a {@link IDTokenClaimsSet}. Configurable calculation strategy
 * function is used for producing the hash.
 */
public class AddAuthorizationCodeHashToIDToken extends AbstractOIDCSigningResponseAction {

    /** Class logger. */
    @Nonnull
    private Logger log = LoggerFactory.getLogger(AddAuthorizationCodeHashToIDToken.class);

    /** The strategy used for calculating the authorization code hash value. */
    @NonnullAfterInit private Function<ProfileRequestContext, String> codeHashCalculationStrategy;

    /**
     * Set the strategy used for calculating the authorization code hash value.
     * 
     * @param strategy calculation strategy
     */
    public void setCodeHashCalculationStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        codeHashCalculationStrategy =
                Constraint.isNotNull(strategy, "Authorization code hash calculation strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (codeHashCalculationStrategy == null) {
            throw new ComponentInitializationException("Authorization code hash calculation strategy cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (getOidcResponseContext().getIDToken() == null) {
            log.error("{} No id token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return;
        }
        final String hashValue = codeHashCalculationStrategy.apply(profileRequestContext);
        if (StringSupport.trimOrNull(hashValue) == null) {
            log.error("{} Could not produce the authorization code hash value", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_SEC_CFG);
            return;
        }
        log.debug("{} Setting authz code hash to id token", getLogPrefix());
        getOidcResponseContext().getIDToken().setClaim(IDTokenClaimsSet.C_HASH_CLAIM_NAME, hashValue);
        log.debug("{} Updated token {}", getLogPrefix(),
                getOidcResponseContext().getIDToken().toJSONObject().toJSONString());

    }

}