/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.text.ParseException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.OutboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.AccessTokenContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.impl.AbstractSignJWTAction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that signs {@link AccessTokenContext#getJWT()} and overwrites it with the signed version.
 * 
 * <p>The action exits gracefully if no signing parameters exist or the JWT does not exist.</p>
 * 
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#MESSAGE_PROC_ERROR}
 * @event {@link EventIds#INVALID_MSG_CTX}
 */
public class SignAccessToken extends AbstractSignJWTAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SignAccessToken.class);

    /** Strategy used to locate the subcontext with the token. */
    @Nonnull private Function<ProfileRequestContext,AccessTokenContext> accessTokenContextLookupStrategy;
    
    /** Token context. */
    @Nullable private AccessTokenContext tokenCtx; 
    
    /** Source of claims set to sign. */
    @Nullable private JWTClaimsSet claimsSet;

    /** Constructor. */
    public SignAccessToken() {
        // PRC -> inbound message context -> OIDC response context -> ATC
        accessTokenContextLookupStrategy = new ChildContextLookup<>(AccessTokenContext.class, true).compose(
                new ChildContextLookup<>(OIDCAuthenticationResponseContext.class).compose(
                        new OutboundMessageContextLookup()));
    }

    /**
     * Set the strategy used to create the {@link AccessTokenContext} to use.
     * 
     * @param strategy creation strategy
     */
    public void setAccessTokenContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,AccessTokenContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        accessTokenContextLookupStrategy =
                Constraint.isNotNull(strategy, "AccessTokenContext creation strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        tokenCtx = accessTokenContextLookupStrategy.apply(profileRequestContext);
        if (tokenCtx == null) {
            log.error("{} AccessTokenContext is missing", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        if (tokenCtx.getJWT() == null) {
            log.debug("{} JWT is absent, nothing to do", getLogPrefix());
            return false;
        }
        
        try {
            claimsSet = tokenCtx.getJWT().getJWTClaimsSet();
        } catch (final ParseException e) {
            log.error("{} Access token JWT parsing failed: {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_PROC_ERROR);
            return false;
        }
        
        return true;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull protected JWTClaimsSet getClaimsSetToSign() {
        return claimsSet;
    }

    /** {@inheritDoc} */
    @Override
    protected void setSignedJWT(@Nonnull final SignedJWT jwt) {
        tokenCtx.setJWT(jwt);
    }

}