/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.OutboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.AccessTokenContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.impl.AbstractOIDCResponseAction;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;

import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that creates a Access Token, and sets it to work context
 * {@link OIDCAuthenticationResponseContext#getAccessToken()} located under
 * {@link ProfileRequestContext#getOutboundMessageContext()}.
 */
public class SetAccessTokenToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetAccessTokenToResponseContext.class);

    /** Strategy used to locate the subcontext with the token. */
    @Nonnull private Function<ProfileRequestContext,AccessTokenContext> accessTokenContextLookupStrategy;
    
    /** Token context. */
    @Nullable private AccessTokenContext tokenCtx; 
    
    /** Constructor. */
    public SetAccessTokenToResponseContext() {
        // PRC -> outbound message context -> OIDC response context -> ATC
        accessTokenContextLookupStrategy = new ChildContextLookup<>(AccessTokenContext.class).compose(
                new ChildContextLookup<>(OIDCAuthenticationResponseContext.class).compose(
                        new OutboundMessageContextLookup()));
    }

    /**
     * Set the strategy used to lookup the {@link AccessTokenContext} to use.
     * 
     * @param strategy lookup strategy
     */
    public void setAccessTokenContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,AccessTokenContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        accessTokenContextLookupStrategy =
                Constraint.isNotNull(strategy, "AccessTokenContext lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        tokenCtx = accessTokenContextLookupStrategy.apply(profileRequestContext);
        if (tokenCtx == null) {
            log.error("{} AccessTokenContext is missing", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        if (tokenCtx.getJWT() == null && tokenCtx.getOpaque() == null) {
            log.debug("{} Access token is missing", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final String token;
        if (tokenCtx.getJWT() != null) {
            token = tokenCtx.getJWT().serialize();
        } else {
            token = tokenCtx.getOpaque();
        }
        
        getOidcResponseContext().setAccessToken(token, tokenCtx.getLifetime(), getOidcResponseContext().getScope());
    }

}