/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.decoding.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.decoder.MessageDecodingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.MoreObjects;
import com.nimbusds.oauth2.sdk.http.HTTPRequest;
import com.nimbusds.oauth2.sdk.http.ServletUtils;
import com.nimbusds.openid.connect.sdk.AuthenticationRequest;

import net.shibboleth.idp.plugin.oidc.op.oauth2.decoding.impl.BaseOAuth2RequestDecoder;

/**
 * Message decoder decoding OpenID Connect {@link AuthenticationRequest}s.
 */
public class OIDCAuthenticationRequestDecoder extends BaseOAuth2RequestDecoder<AuthenticationRequest> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(OIDCAuthenticationRequestDecoder.class);

    /** {@inheritDoc} */
    @Override
    protected AuthenticationRequest parseMessage() throws MessageDecodingException {
        try {
            final HTTPRequest httpReq = ServletUtils.createHTTPRequest(getHttpServletRequest());
            getProtocolMessageLog().trace("Inbound request {}", RequestUtil.toString(httpReq));
            return AuthenticationRequest.parse(httpReq);
        } catch (final com.nimbusds.oauth2.sdk.ParseException | IOException e) {
            log.error("Unable to decode inbound request: {}", e.getMessage());
            throw new MessageDecodingException(e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected String getMessageToLog(@Nullable final AuthenticationRequest message) {
        return message == null ? null : MoreObjects.toStringHelper(this).omitNullValues()
                .add("maxAge", message.getMaxAge())
                .add("acrValues", message.getACRValues())
                .add("claimsLocales", message.getClaimsLocales())
                .add("clientId", message.getClientID())
                .add("codeChallenge", message.getCodeChallenge())
                .add("codeChallengeMethod", message.getCodeChallengeMethod())
                .add("customParameters", message.getCustomParameters())
                .add("display", message.getDisplay())
                .add("endpointURI", message.getEndpointURI())
                .add("idTokenHint", message.getIDTokenHint() == null ? null : message.getIDTokenHint().serialize())
                .add("loginHint", message.getLoginHint())
                .add("nonce", message.getNonce())
                .add("oidcClaims", message.getOIDCClaims())
                .add("prompt", message.getPrompt())
                .add("purpose", message.getPurpose())
                .add("redirectionURI", message.getRedirectionURI())
                .add("requestObject", message.getRequestObject() == null ?
                        null : message.getRequestObject().serialize())
                .add("requestURI", message.getRequestURI())
                .add("resources", message.getResources())
                .add("responseMode", message.getResponseMode())
                .add("responseType", message.getResponseType())
                .add("scope", message.getScope())
                .add("state", message.getState())
                .add("uiLocales", message.getUILocales())
                .toString();
    }

}