/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.authn.impl;

import java.util.Collections;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.oidc.authn.context.OAuth2ClientAuthenticationContext;
import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.oidc.profile.config.navigate.TokenEndpointAuthMethodLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Validates the client authentication type with the token_endpoint_auth_method stored in the client's metadata
 * and the profile configuration.
 * 
 * <p>In the absence of metadata, the profile configuration is used alone.</p>
 * 
 * @pre {@link OIDCMetadataContext} is available
 * @pre AuthenticationContext.getSubcontext(OAuth2ClientAuthenticationContext.class) != null
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#ACCESS_DENIED}
 */
public class ValidateClientAuthenticationType extends AbstractAuthenticationAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ValidateClientAuthenticationType.class);
    
    /** Strategy that will return {@link OIDCMetadataContext}. */
    @Nonnull private Function<ProfileRequestContext,OIDCMetadataContext> oidcMetadataContextLookupStrategy;
    
    /** Strategy to obtain enabled token endpoint authentication methods. */
    @Nullable private Function<ProfileRequestContext,Set<ClientAuthenticationMethod>> 
        tokenEndpointAuthMethodsLookupStrategy;
    
    /** The attached OIDC metadata context. */
    @Nullable private OIDCMetadataContext oidcMetadataContext;

    /** The extracted client authentication information. */
    @Nullable private ClientAuthentication clientAuthentication;
    
    /** Enabled client authn methods. */
    @Nullable @NonnullElements private Set<ClientAuthenticationMethod> enabledMethods;
    
    /**
     * Constructor.
     */
    public ValidateClientAuthenticationType() {
        oidcMetadataContextLookupStrategy = new ChildContextLookup<>(OIDCMetadataContext.class).compose(
                new InboundMessageContextLookup());
        tokenEndpointAuthMethodsLookupStrategy = new TokenEndpointAuthMethodLookupFunction();
    }
        
    /**
     * Set the strategy used to return the {@link OIDCMetadataContext}.
     * 
     * @param strategy The lookup strategy.
     */
    public void setOIDCMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCMetadataContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        oidcMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, "OIDCMetadataContext lookup strategy cannot be null");
    }
    
    /**
     * Set strategy to obtain enabled token endpoint authentication methods.
     * @param strategy What to set.
     */
    public void setTokenEndpointAuthMethodsLookupStrategy(@Nonnull final Function<ProfileRequestContext, 
            Set<ClientAuthenticationMethod>> strategy) {
        tokenEndpointAuthMethodsLookupStrategy = Constraint.isNotNull(strategy, 
                "Strategy to obtain enabled token endpoint authentication methods cannot be null");
        
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {
        if (!super.doPreExecute(profileRequestContext, authenticationContext)) {
            return false;
        }

        final OAuth2ClientAuthenticationContext oauth2Ctx =
                authenticationContext.getSubcontext(OAuth2ClientAuthenticationContext.class);
        if (oauth2Ctx != null) {
            clientAuthentication = oauth2Ctx.getClientAuthentication();
        }

        oidcMetadataContext = oidcMetadataContextLookupStrategy.apply(profileRequestContext);
        
        enabledMethods = tokenEndpointAuthMethodsLookupStrategy.apply(profileRequestContext);
        if (enabledMethods == null) {
            enabledMethods = Collections.emptySet();
        }
        
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {
        
        final ClientAuthenticationMethod registeredMethod;

        // Pull the client's registered authn method, or default to client_secret_basic.
        // If no metadata exists, leave null.
        if (oidcMetadataContext != null && oidcMetadataContext.getClientInformation() != null) {
            final OIDCClientMetadata clientMetadata = oidcMetadataContext.getClientInformation().getOIDCMetadata();
            registeredMethod = clientMetadata.getTokenEndpointAuthMethod() != null ? 
                    clientMetadata.getTokenEndpointAuthMethod() : ClientAuthenticationMethod.CLIENT_SECRET_BASIC;
        } else {
            registeredMethod = null;
        }

        // Did the client use what it registered and is that still allowed?
        // The enabledMethods member contains the methods authorized in the configuration as a whole.

        final ClientAuthenticationMethod used =
                clientAuthentication != null ? clientAuthentication.getMethod() : ClientAuthenticationMethod.NONE;

        if (registeredMethod != null && !registeredMethod.equals(used)) {
            log.warn("{} Client '{}' registered {} but attempted {}", getLogPrefix(), getClientID(),
                    registeredMethod, used);
            ActionSupport.buildEvent(profileRequestContext, EventIds.ACCESS_DENIED);
        } else if (!enabledMethods.contains(used)) {
            log.warn("{} Requested method {} not enabled in profile configuration", getLogPrefix(), used);
            ActionSupport.buildEvent(profileRequestContext, EventIds.ACCESS_DENIED);
        }
    }

    /**
     * Parses the client ID from OIDC metadata or client authentication, if exists.
     * 
     * @return client ID, or null it it couldn't be found.
     */
    @Nullable private String getClientID() {
        if (oidcMetadataContext != null && oidcMetadataContext.getClientInformation() != null) {
            return oidcMetadataContext.getClientInformation().getID().getValue();
        }
        if (clientAuthentication != null && clientAuthentication.getClientID() != null) {
            return clientAuthentication.getClientID().getValue();
        }
        return null;
    }
}