/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import javax.annotation.Nonnull;

import net.shibboleth.idp.authn.context.SubjectContext;
import net.shibboleth.oidc.profile.core.OidcEventIds;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An action that creates an {@link SubjectContext} and attaches it to the current {@link ProfileRequestContext}.
 * The principal is set by the information provided by the Authorization Code / Access Token.
 */
public class InitializeSubjectContext extends AbstractOIDCTokenResponseAction {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(InitializeSubjectContext.class);

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        if (getOidcResponseContext().getAuthorizationGrantClaimsSet() == null) {
            log.warn("{} Subject principal not resolved from prior authorization grant", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_SUBJECT);
            return false;
        }
        
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final SubjectContext subCtx = new SubjectContext();
        subCtx.setPrincipalName(getOidcResponseContext().getAuthorizationGrantClaimsSet().getPrincipal());
        
        // Replace existing context. Generally there will already be a prior context representing
        // authentication of the client, so this replacement happens when the client is not the
        // subject of the authorization grant and claims.
        profileRequestContext.addSubcontext(subCtx, true);
        
        log.debug("{} Created subject context for principal '{}'", getLogPrefix(), subCtx.getPrincipalName());
    }

}