/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.metadata.impl;

import java.io.IOException;
import java.util.List;
import java.util.Timer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.springframework.core.io.Resource;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.id.Identifier;

import net.minidev.json.JSONValue;
import net.shibboleth.idp.plugin.oidc.op.metadata.resolver.RefreshableMetadataValueResolver;
import net.shibboleth.oidc.metadata.impl.AbstractFileOIDCEntityResolver;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * An implementation to {@link RefreshableMetadataValueResolver} that fetches the information from a file.
 */
public class FilesystemMetadataValueResolver extends AbstractFileOIDCEntityResolver<Identifier, Object> 
    implements RefreshableMetadataValueResolver {

    /**
     * Constructor.
     * 
     * @param metadata the metadata file
     * 
     * @throws IOException If the metedata cannot be loaded.
     */
    public FilesystemMetadataValueResolver(@Nonnull final Resource metadata) throws IOException {
        super(metadata);
    }
    
    /**
     * Constructor.
     * 
     * @param metadata the metadata file
     * @param backgroundTaskTimer timer used to refresh metadata in the background
     * 
     * @throws IOException If the metedata cannot be loaded.
     */
    public FilesystemMetadataValueResolver(@Nullable final Timer backgroundTaskTimer, 
            @Nonnull final Resource metadata) throws IOException {
        super(backgroundTaskTimer, metadata);
    }

    /**
     * Returns all the resolved objects.
     * 
     * @param profileRequestContext the criteria to evaluate or process, currently ignored. May be null.
     * 
     * @return all the resolved objects.
     * 
     * @throws ResolverException thrown if there is an error during resolution.
     */
    @Override
    @Nullable public Iterable<Object> resolve(@Nullable final ProfileRequestContext profileRequestContext)
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        return getBackingStore().getOrderedInformation();
    }

    /**
     * Returns a single resolved object. If many were resolved, a single one is selected randomly from the set.
     * 
     * @param profileRequestContext the criteria to evaluate or process, currently ignored. May be null.
     * 
     * @return a single resolved object.
     * 
     * @throws ResolverException thrown if there is an error during resolution.
     */
    @Override
    @Nullable public Object resolveSingle(@Nullable final ProfileRequestContext profileRequestContext)
            throws ResolverException {
        return resolve(profileRequestContext).iterator().next();
    }

    /** {@inheritDoc} */
    @Override
    protected List<Object> parse(final byte[] bytes) throws ParseException {
        return List.of(JSONValue.parse(bytes));
    }

    /** {@inheritDoc} */
    @Override
    protected Identifier getKey(final Object value) {
        return new Identifier("dynamic metadata value");
    }

}
