/*===========================================================================
  Copyright (C) 2008-2011 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
===========================================================================*/

package net.sf.okapi.common;

import net.sf.okapi.common.exceptions.OkapiUnexpectedResourceTypeException;
import net.sf.okapi.common.resource.DocumentPart;
import net.sf.okapi.common.resource.EndSubfilter;
import net.sf.okapi.common.resource.Ending;
import net.sf.okapi.common.resource.ITextUnit;
import net.sf.okapi.common.resource.MultiEvent;
import net.sf.okapi.common.resource.PipelineParameters;
import net.sf.okapi.common.resource.RawDocument;
import net.sf.okapi.common.resource.StartDocument;
import net.sf.okapi.common.resource.StartGroup;
import net.sf.okapi.common.resource.StartSubDocument;
import net.sf.okapi.common.resource.StartSubfilter;

/**
 * Represents an event generated by a filter that implements IFilter.
 */
public class Event {

	/* All of these events are technically a singletons.
	 * Reusing these is dangerous, since any step can attach annotations on them.
	 * Imagine pipelines running in parallel and annotating these global objects. 
	 */

	/** @deprecated Use {@link #createEndBatchEvent()} instead of this global object. */
	@Deprecated
	public final static Event END_BATCH_EVENT = new Event(EventType.END_BATCH);

	/** @deprecated Use {@link #createNoopEvent()} instead of this global object. */
	@Deprecated
	public final static Event NOOP_EVENT = new Event(EventType.NO_OP);

	/** @deprecated Use {@link #createStartBatchEvent()} instead of this global object. */
	@Deprecated
	public final static Event START_BATCH_EVENT = new Event(EventType.START_BATCH);

	/** @deprecated Never used in Okapi, so we will not create a helper method. But using it is dangerous. */
	@Deprecated
	public final static Event START_BATCH_ITEM_EVENT = new Event(EventType.START_BATCH_ITEM);

	private EventType filterEventType;
	private IResource resource;
	
	public Event() {
	}

	/**
	 * Creates a new event without any associated resource. Used for filter events that have no resources such as START
	 * and FINISH.
	 * 
	 * @param filterEventType
	 *            the type of event to create.
	 */
	public Event(EventType filterEventType) {
		this.filterEventType = filterEventType;
	}

	/**
	 * Creates a new event with an associated resource.
	 * 
	 * @param filterEventType
	 *            the type of event to create.
	 * @param resource
	 *            the resource to associate to the event.
	 */
	public Event(EventType filterEventType, IResource resource) {
		this.filterEventType = filterEventType;
		this.resource = resource;
	}

	/**
	 * Creates a new event with an associated resource and a skeleton object.
	 * 
	 * @param filterEventType
	 *            the type of event to create.
	 * @param resource
	 *            the resource to associate to the event.
	 * @param skeleton
	 *            the skeleton to associate to the event.
	 */
	public Event(EventType filterEventType,
		IResource resource,
		ISkeleton skeleton)
	{
		this.filterEventType = filterEventType;
		this.resource = resource;
		this.resource.setSkeleton(skeleton);
	}

	/**
	 * Gets the type of this event.
	 * 
	 * @return the type of this event.
	 */
	public EventType getEventType () {
		return filterEventType;
	}

	/**
	 * Gets the resource associated to this event.
	 * 
	 * @return the resource associated to this event, or null if there is none.
	 */
	public IResource getResource () {
		return resource;
	}

	/**
	 * Sets the resource associated to this event.
	 * @param resource the new resource.
	 */
	public void setResource (IResource resource) {
		this.resource = resource;
	}

	public void setEventType(EventType filterEventType) {
		this.filterEventType = filterEventType;
	}

	/**
	 * Convenience method to tell if this Event carries a {@link ITextUnit}
	 * 
	 * @return true if {@link ITextUnit}, false otherwise
	 */
	public boolean isTextUnit () {
		return (filterEventType == EventType.TEXT_UNIT);
	}

	/**
	 * Convenience method to tell if this Event carries a {@link DocumentPart}
	 * 
	 * @return true if {@link DocumentPart}, false otherwise
	 */
	public boolean isDocumentPart () {
		return (filterEventType == EventType.DOCUMENT_PART);
	}

	/**
	 * Convenience method to tell if this Event carries a {@link StartGroup}
	 * 
	 * @return true if {@link StartGroup}, false otherwise
	 */
	public boolean isStartGroup () {
		return (filterEventType == EventType.START_GROUP);
	}

	/**
	 * Convenience method to tell if this Event carries a group {@link Ending}
	 * 
	 * @return true if group {@link Ending}, false otherwise
	 */
	public boolean isEndGroup () {
		return (filterEventType == EventType.END_GROUP);
	}

	/**
	 * Convenience method to tell if this Event carries a {@link RawDocument}
	 * 
	 * @return true if {@link RawDocument}, false otherwise
	 */
	public boolean isRawDocument () {
		return (filterEventType == EventType.RAW_DOCUMENT);
	}
	
	/**
	 * Convenience method to tell if this Event carries a {@link StartDocument}
	 * 
	 * @return true if {@link StartDocument}, false otherwise
	 */
	public boolean isStartDocument () {
		return (filterEventType == EventType.START_DOCUMENT);
	}
	
	/**
	 * Convenience method to tell if this Event carries a document {@link Ending}
	 * 
	 * @return true if document {@link Ending}, false otherwise
	 */
	public boolean isEndDocument () {
		return (filterEventType == EventType.END_DOCUMENT);
	}
	
	/**
	 * Convenience method to tell if this Event is START_BATCH_ITEM 
	 * 
	 * @return true if START_BATCH_ITEM, false otherwise
	 */
	public boolean isStartBatchItem () {
		return (filterEventType == EventType.START_BATCH_ITEM);
	}
	
	/**
	 * Convenience method to tell if this Event carries a {@link StartSubDocument}
	 * 
	 * @return true if {@link StartSubDocument}, false otherwise
	 */
	public boolean isStartSubDocument () {
		return (filterEventType == EventType.START_SUBDOCUMENT);
	}
	
	/**
	 * Convenience method to tell if this Event carries a sub-document {@link Ending}
	 * 
	 * @return true if sub-document {@link Ending}, false otherwise
	 */
	public boolean isEndSubDocument () {
		return (filterEventType == EventType.END_SUBDOCUMENT);
	}

	/**
	 * Convenience method to tell if this Event is a MULTI_EVENT
	 * 
	 * @return true if {@link MultiEvent}, false otherwise
	 */
	public boolean isMultiEvent () {
		return (filterEventType == EventType.MULTI_EVENT);
	}
	
	/**
	 * Convenience method to tell if this Event is a SubFilter Group
	 * 
	 * @return true if {@link StartSubfilter}, false otherwise
	 */
	public boolean isStartSubfilter () {
		return (filterEventType == EventType.START_SUBFILTER);
	}
	
	/**
	 * Convenience method to tell if this Event is a SubFilter Group
	 * 
	 * @return true if {@link StartSubfilter}, false otherwise
	 */
	public boolean isEndSubfilter () {
		return (filterEventType == EventType.END_SUBFILTER);
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link ITextUnit}.
	 * The caller should confirm the {@link Event} type using isTextUnit before
	 * calling this method.
	 * 
	 * @return the {@link ITextUnit}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link ITextUnit}
	 */
	public ITextUnit getTextUnit () {
		if ( isTextUnit() ) {
			return (ITextUnit)resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a text unit.");
	}

	/**
	 * Convenience method returns the {@link IResource} as a {@link DocumentPart}.
	 * The caller should confirm the {@link Event} type using isDocumentPart before calling
	 * this method.
	 * 
	 * @return the {@link DocumentPart}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 * 		if the {@link IResource} is not a {@link DocumentPart}
	 */
	public DocumentPart getDocumentPart () {
		if ( isDocumentPart() ) {
			return (DocumentPart) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a DocumentPart");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link StartGroup}. The 
	 * caller should confirm the {@link Event} type using isStartGroup before calling this
	 * method.
	 * 
	 * @return the {@link StartGroup}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link StartGroup}
	 */
	public StartGroup getStartGroup () {
		if ( isStartGroup() ) {
			return (StartGroup) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a StartGroup");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link Ending}. The caller
	 * should confirm the {@link Event} type using isEndGroup before calling this method.
	 * 
	 * @return the {@link Ending}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link Ending}
	 */
	public Ending getEndGroup () {
		if ( isEndGroup() ) {
			return (Ending) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not an Ending");
	}
	
	/**
	 * Convenience method returns the Ending resource.
	 * @return the Ending resource.
	 */
	public Ending getEnding () {
		return (Ending)resource;
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link RawDocument}. The
	 * caller should confirm the {@link Event} type using isRawDocument before calling
	 * this method.
	 * 
	 * @return the {@link RawDocument}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link RawDocument}
	 */
	public RawDocument getRawDocument() {
		if ( isRawDocument() ) {
			return (RawDocument) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a RawDocument");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link StartSubDocument}. The
	 * caller should confirm the {@link Event} type using isStartSubDocument before calling
	 * this method.
	 * 
	 * @return the {@link StartSubDocument}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link StartSubDocument}
	 */
	public StartSubDocument getStartSubDocument() {
		if ( isStartSubDocument() ) {
			return (StartSubDocument) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a StartSubDocument");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link StartDocument}. The
	 * caller should confirm the {@link Event} type using isStartDocument before calling
	 * this method.
	 * 
	 * @return the {@link StartDocument}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link StartDocument}
	 */
	public StartDocument getStartDocument() {
		if ( isStartDocument() ) {
			return (StartDocument) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a StartDocument");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link StartSubfilter}. The
	 * caller should confirm the {@link Event} type using isStartSubfilter before calling
	 * this method.
	 * 
	 * @return the {@link StartSubfilter}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link StartSubfilter}
	 */
	public StartSubfilter getStartSubfilter() {
		if ( isStartSubfilter() ) {
			return (StartSubfilter)resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a StartSubfilter");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link EndSubfilter}. The
	 * caller should confirm the {@link Event} type using isEndSubfilter before calling
	 * this method.
	 * 
	 * @return the {@link EndSubfilter}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link EndSubfilter}
	 */
	public EndSubfilter getEndSubfilter() {
		if ( isEndSubfilter() ) {
			return (EndSubfilter)resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a EndSubfilter");
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link MultiEvent}. The
	 * caller should confirm the {@link EventType} using isMultiEvent before calling
	 * this method.
	 * 
	 * @return the {@link MultiEvent}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException
	 *             if the {@link IResource} is not a {@link MultiEvent}
	 */
	public MultiEvent getMultiEvent() {
		if ( isMultiEvent() ) {
			return (MultiEvent) resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a MultiEvent");
	}
	
	/**
	 * Indicates if this event carries a {@link PipelineParameters} resource.
	 * @return true if this event carries a {@link PipelineParameters} resource, false otherwise.
	 */
	public boolean isPipelineParametersEvent () {
		return (filterEventType == EventType.PIPELINE_PARAMETERS);
	}
	
	/**
	 * Convenience method returns the {@link IResource} as a {@link PipelineParameters}.
	 * 
	 * @return the {@link PipelineParameters}
	 * 
	 * @throws OkapiUnexpectedResourceTypeException 
	 * 		if the {@link IResource} is not a {@link PipelineParameters}
	 */
	public PipelineParameters getPipelineParameters () {
		if ( isPipelineParametersEvent() ) {
			return (PipelineParameters)resource;
		}
		throw new OkapiUnexpectedResourceTypeException("Event resource is not a PipelineParameters");
	}
	
	@Override
	public String toString() {
		return filterEventType.toString();
	}

	public boolean isNoop() {
		if (filterEventType == EventType.NO_OP) {
			return true;
		}		
		return false;
	}

	/** Convenience method that creates an {@link EventType#END_BATCH} event */
	public final static Event createEndBatchEvent() {
		return new Event(EventType.END_BATCH);
	}

	/** Convenience method that creates an {@link EventType#NO_OP} event */
	public final static Event createNoopEvent() {
		return new Event(EventType.NO_OP);
	}

	/** Convenience method that creates an {@link EventType#START_BATCH} event */
	public final static Event createStartBatchEvent() {
		return new Event(EventType.START_BATCH);
	}
}
