package net.sf.jagg;

/**
 * This class represents all statistics generated by a
 * <code>LinearRegressionAggregator</code>.
 *
 * @author Randy Gettman
 * @since 0.1.0
 *
 * @see LinearRegressionAggregator
 */
public class LinearRegressionStats
{
   private double myLineSlope;
   private double myLineIntercept;
   private long myCount;
   private double myRSquared;
   private double myCorrelation;
   private double myAvg1;
   private double myAvg2;

   /**
    * Creates a <code>LinearRegressionStats</code> with the given attributes.
    *
    * @param lineSlope The linear regression line's slope.
    * @param lineIntercept The linear regression line's y-intercept.
    * @param count The number of non-null samples considered.
    * @param rSquared The coefficient of determination, a.k.a. R-squared, or goodness
    *    of fit.
    * @param correlation The coefficient of correlation between the first "x"
    *    property and the second "y" property.
    * @param avg1 The average of the first "x" property.
    * @param avg2 The average of the second "y" property.
    */
   public LinearRegressionStats(double lineSlope, double lineIntercept, long count,
      double rSquared, double correlation, double avg1, double avg2)
   {
      myLineSlope = lineSlope;
      myLineIntercept = lineIntercept;
      myCount = count;
      myRSquared = rSquared;
      myCorrelation = correlation;
      myAvg1 = avg1;
      myAvg2 = avg2;
   }

   // Getters.
   /**
    * Returns the linear regression line's slope.
    * @return The linear regression line's slope.
    */
   public double getLineSlope()
   {
      return myLineSlope;
   }

   /**
    * Returns the linear regression line's y-intercept.
    * @return The linear regression line's y-intercept.
    */
   public double getLineIntercept()
   {
      return myLineIntercept;
   }

   /**
    * Returns the number of non-null samples considered.
    * @return The number of non-null samples considered.
    */
   public long getCount()
   {
      return myCount;
   }

   /**
    * Returns the coefficient of determination, a.k.a. R-squared, or goodness
    * of fit.
    * @return The coefficient of determination.
    */
   public double getRSquared()
   {
      return myRSquared;
   }

   /**
    * Returns the coefficient of correlation between the first "x" property and
    * the second "y" property.
    * @return The coefficient of correlation.
    */
   public double getCorrelation()
   {
      return myCorrelation;
   }

   /**
    * Returns the average of the first "x" property.
    * @return The average of the first "x" property.
    */
   public double getAvg1()
   {
      return myAvg1;
   }

   /**
    * Returns the average of the second "y" property.
    * @return The average of the second "y" property.
    */
   public double getAvg2()
   {
      return myAvg2;
   }

   /**
    * The <code>String</code> representation: <br>
    * <code>(slope,intercept,count,rSquared,correlation,avg1,avg2)</code>.
    *
    * @return A <code>String</code> representation.
    */
   public String toString()
   {
      StringBuilder buf = new StringBuilder();
      buf.append("(");
      buf.append(getLineSlope());
      buf.append(",");
      buf.append(getLineIntercept());
      buf.append(",");
      buf.append(getCount());
      buf.append(",");
      buf.append(getRSquared());
      buf.append(",");
      buf.append(getCorrelation());
      buf.append(",");
      buf.append(getAvg1());
      buf.append(",");
      buf.append(getAvg2());
      buf.append(")");
      return buf.toString();
   }
}
