/*
 * Decompiled with CFR 0.152.
 */
package com.terracottatech.offheapstore.paging;

import com.terracottatech.offheapstore.buffersource.BufferSource;
import com.terracottatech.offheapstore.paging.OffHeapStorageArea;
import com.terracottatech.offheapstore.paging.Page;
import com.terracottatech.offheapstore.paging.PageSource;
import com.terracottatech.offheapstore.storage.allocator.PowerOfTwoAllocator;
import com.terracottatech.offheapstore.util.DebuggingUtils;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UpfrontAllocatingPageSource
implements PageSource {
    private static final Logger LOGGER = LoggerFactory.getLogger(UpfrontAllocatingPageSource.class);
    private static final int ALLOCATION_PROGRESS_INTERVAL = 10;
    private static final long ALLOCATION_PROGRESS_STEP_BYTES = 0x100000000L;
    private static final Comparator<Page> REGION_COMPARATOR = new Comparator<Page>(){

        @Override
        public int compare(Page a, Page b) {
            if (a.address() == b.address()) {
                return a.size() - b.size();
            }
            return a.address() - b.address();
        }
    };
    private final List<PowerOfTwoAllocator> sliceAllocators = new ArrayList<PowerOfTwoAllocator>();
    private final List<PowerOfTwoAllocator> victimAllocators = new ArrayList<PowerOfTwoAllocator>();
    private final List<ByteBuffer> buffers = new ArrayList<ByteBuffer>();
    private final List<SortedSet<Page>> victims = new ArrayList<SortedSet<Page>>();
    private volatile int availableSet = -1;

    public UpfrontAllocatingPageSource(BufferSource source, long max, int chunk) {
        this(source, max, chunk, -1, true);
    }

    public UpfrontAllocatingPageSource(BufferSource source, long max, int maxChunk, int minChunk) {
        this(source, max, maxChunk, minChunk, false);
    }

    private UpfrontAllocatingPageSource(BufferSource source, long max, int maxChunk, int minChunk, boolean fixed) {
        LOGGER.info("Allocating {}B in chunks", (Object)DebuggingUtils.toBase2SuffixedString(max));
        long start = System.currentTimeMillis();
        int chunk = (int)Math.min(max, (long)maxChunk);
        long size = 0L;
        int print = 10;
        int previousStep = 0;
        long i = 0L;
        while (i < max) {
            int chunkSize = (int)Math.min((long)chunk, max - i);
            ByteBuffer b = source.allocateBuffer(chunkSize);
            if (b == null) {
                LOGGER.info("Allocated {}B in {}B chunks.", new Object[]{DebuggingUtils.toBase2SuffixedString(size), DebuggingUtils.toBase2SuffixedString(chunk)});
                size = 0L;
                if (!fixed && (chunk >>>= 1) >= minChunk) continue;
                throw new IllegalArgumentException("An attempt was made to allocate more off-heap memory than the JVM can allow. The limit on off-heap memory size is given by the -XX:MaxDirectMemorySize command (or equivalent).");
            }
            this.sliceAllocators.add(new PowerOfTwoAllocator(chunkSize));
            this.victimAllocators.add(new PowerOfTwoAllocator(chunkSize));
            this.victims.add(new TreeSet<Page>(REGION_COMPARATOR));
            this.buffers.add(b);
            long percentAllocated = 100L * (i += (long)chunkSize) / max;
            long step = (size += (long)chunkSize) / 0x100000000L;
            if (percentAllocated >= (long)print) {
                print += 10;
                if (step > (long)previousStep && percentAllocated < 100L) {
                    LOGGER.info("Allocation {}% complete", (Object)(print - 10));
                    ++previousStep;
                }
            }
            if (percentAllocated != 100L) continue;
            LOGGER.info("Allocation 100% complete");
        }
        long end = System.currentTimeMillis();
        long timeToAllocate = end - start;
        LOGGER.info("Allocated {}B in {}B chunks.", new Object[]{DebuggingUtils.toBase2SuffixedString(size), DebuggingUtils.toBase2SuffixedString(chunk)});
        LOGGER.info("Took {} ms to create CacheManager off-heap storage of {}B.", (Object)timeToAllocate, (Object)DebuggingUtils.toBase2SuffixedString(max));
    }

    @Override
    public Page allocate(int size, boolean thief, boolean victim, OffHeapStorageArea owner) {
        if (thief) {
            return this.allocateAsThief(size, victim, owner);
        }
        return this.allocateFromFree(size, victim, owner);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Page allocateAsThief(int size, boolean victim, OffHeapStorageArea owner) {
        Page free = this.allocateFromFree(size, victim, owner);
        if (free != null) {
            return free;
        }
        PowerOfTwoAllocator victimAllocator = null;
        PowerOfTwoAllocator sliceAllocator = null;
        List<Object> targets = Collections.emptyList();
        ArrayList<AllocatedRegion> tempHolds = new ArrayList<AllocatedRegion>();
        IdentityHashMap releases = new IdentityHashMap();
        UpfrontAllocatingPageSource upfrontAllocatingPageSource = this;
        synchronized (upfrontAllocatingPageSource) {
            for (int i = 0; i < this.victimAllocators.size(); ++i) {
                int n = this.victimAllocators.get(i).find(size);
                if (n < 0) continue;
                victimAllocator = this.victimAllocators.get(i);
                sliceAllocator = this.sliceAllocators.get(i);
                targets = this.findVictimPages(i, n, size);
                int claimAddress = n;
                for (Page page : targets) {
                    victimAllocator.claim(page.address(), page.size());
                    int claimSize = page.address() - claimAddress;
                    if (claimSize > 0) {
                        tempHolds.add(new AllocatedRegion(claimAddress, claimSize));
                        sliceAllocator.claim(claimAddress, claimSize);
                        victimAllocator.claim(claimAddress, claimSize);
                    }
                    claimAddress = page.address() + page.size();
                }
                int n2 = n + size - claimAddress;
                if (n2 <= 0) break;
                tempHolds.add(new AllocatedRegion(claimAddress, n2));
                sliceAllocator.claim(claimAddress, n2);
                victimAllocator.claim(claimAddress, n2);
                break;
            }
            for (Page page : targets) {
                OffHeapStorageArea a = page.binding();
                Collection collection = (Collection)releases.get(a);
                if (collection == null) {
                    LinkedList<Page> linkedList = new LinkedList<Page>();
                    linkedList.add(page);
                    releases.put(a, linkedList);
                    continue;
                }
                collection.add(page);
            }
        }
        LinkedList<Page> results = new LinkedList<Page>();
        for (Map.Entry entry : releases.entrySet()) {
            OffHeapStorageArea a = (OffHeapStorageArea)entry.getKey();
            Collection collection = (Collection)entry.getValue();
            results.addAll(a.release(collection));
        }
        ArrayList<Page> failedReleases = new ArrayList<Page>();
        UpfrontAllocatingPageSource object = this;
        synchronized (object) {
            for (AllocatedRegion allocatedRegion : tempHolds) {
                sliceAllocator.free(allocatedRegion.address, allocatedRegion.size);
                victimAllocator.free(allocatedRegion.address, allocatedRegion.size);
            }
            if (results.size() == targets.size()) {
                for (Page page : targets) {
                    victimAllocator.free(page.address(), page.size());
                    this.free(page);
                }
                return this.allocateFromFree(size, victim, owner);
            }
            for (Page page : targets) {
                if (results.contains(page)) {
                    victimAllocator.free(page.address(), page.size());
                    this.free(page);
                    continue;
                }
                failedReleases.add(page);
            }
        }
        try {
            Page page = this.allocateAsThief(size, victim, owner);
            return page;
        }
        finally {
            UpfrontAllocatingPageSource upfrontAllocatingPageSource2 = this;
            synchronized (upfrontAllocatingPageSource2) {
                for (Page page : failedReleases) {
                    if (!this.victims.get(page.index()).contains(page)) continue;
                    victimAllocator.tryFree(page.address(), page.size());
                }
            }
        }
    }

    private List<Page> findVictimPages(int chunk, int address, int size) {
        return new ArrayList<Page>(this.victims.get(chunk).subSet(new Page(null, -1, address, null), new Page(null, -1, address + size, null)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Page allocateFromFree(int size, boolean victim, OffHeapStorageArea owner) {
        if (Integer.bitCount(size) != 1) {
            int rounded = Integer.highestOneBit(size) << 1;
            LOGGER.info("Request to allocate {}B will allocate {}B", (Object)size, (Object)DebuggingUtils.toBase2SuffixedString(rounded));
            size = rounded;
        }
        if (this.isUnavailable(size)) {
            return null;
        }
        UpfrontAllocatingPageSource upfrontAllocatingPageSource = this;
        synchronized (upfrontAllocatingPageSource) {
            for (int i = 0; i < this.sliceAllocators.size(); ++i) {
                int address = this.sliceAllocators.get(i).allocate(size);
                if (address < 0) continue;
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Allocating a {}B buffer from chunk {} &{}", new Object[]{DebuggingUtils.toBase2SuffixedString(size), i, address});
                }
                ByteBuffer b = ((ByteBuffer)this.buffers.get(i).limit(address + size).position(address)).slice();
                Page p = new Page(b, i, address, owner);
                if (victim) {
                    this.victims.get(i).add(p);
                } else {
                    this.victimAllocators.get(i).claim(address, size);
                }
                return p;
            }
            this.markUnavailable(size);
            return null;
        }
    }

    @Override
    public synchronized void free(Page page) {
        if (page.isFreeable()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Freeing a {}B buffer from chunk {} &{}", new Object[]{DebuggingUtils.toBase2SuffixedString(page.size()), page.index(), page.address()});
            }
            this.markAllAvailable();
            this.sliceAllocators.get(page.index()).free(page.address(), page.size());
            this.victims.get(page.index()).remove(page);
            this.victimAllocators.get(page.index()).tryFree(page.address(), page.size());
        }
    }

    public synchronized long getAllocatedSize() {
        long sum = 0L;
        for (PowerOfTwoAllocator a : this.sliceAllocators) {
            sum += (long)a.occupied();
        }
        return sum;
    }

    public long getAllocatedSizeUnSync() {
        long sum = 0L;
        for (PowerOfTwoAllocator a : this.sliceAllocators) {
            sum += (long)a.occupied();
        }
        return sum;
    }

    private boolean isUnavailable(int size) {
        return (this.availableSet & size) == 0;
    }

    private synchronized void markAllAvailable() {
        this.availableSet = -1;
    }

    private synchronized void markUnavailable(int size) {
        this.availableSet &= ~size;
    }

    public synchronized String toString() {
        StringBuilder sb = new StringBuilder("UpfrontAllocatingPageSource");
        for (int i = 0; i < this.buffers.size(); ++i) {
            sb.append("\nChunk ").append(i + 1).append('\n');
            sb.append("Size             : ").append(DebuggingUtils.toBase2SuffixedString(this.buffers.get(i).capacity())).append("B\n");
            sb.append("Free Allocator   : ").append(this.sliceAllocators.get(i)).append('\n');
            sb.append("Victim Allocator : ").append(this.victimAllocators.get(i));
        }
        return sb.toString();
    }

    static class AllocatedRegion {
        private final int address;
        private final int size;

        AllocatedRegion(int address, int size) {
            this.address = address;
            this.size = size;
        }
    }
}

