package net.jqwik.time.internal.properties.configurators;

import java.time.*;
import java.time.temporal.*;

import net.jqwik.api.*;
import net.jqwik.api.configurators.*;
import net.jqwik.api.providers.*;
import net.jqwik.time.api.arbitraries.*;
import net.jqwik.time.api.constraints.*;
import net.jqwik.time.internal.properties.arbitraries.*;

public class PrecisionConfigurator {

	public static class ForLocalDateTime extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(LocalDateTime.class);
		}

		public Arbitrary<LocalDateTime> configure(Arbitrary<LocalDateTime> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof LocalDateTimeArbitrary) {
				LocalDateTimeArbitrary localDateTimeArbitrary = (LocalDateTimeArbitrary) arbitrary;
				return localDateTimeArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForInstant extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(Instant.class);
		}

		public Arbitrary<Instant> configure(Arbitrary<Instant> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof InstantArbitrary) {
				InstantArbitrary instantArbitrary = (InstantArbitrary) arbitrary;
				return instantArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForOffsetDateTime extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(OffsetDateTime.class);
		}

		public Arbitrary<OffsetDateTime> configure(Arbitrary<OffsetDateTime> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof OffsetDateTimeArbitrary) {
				OffsetDateTimeArbitrary offsetDateTimeArbitrary = (OffsetDateTimeArbitrary) arbitrary;
				return offsetDateTimeArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForZonedDateTime extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(ZonedDateTime.class);
		}

		public Arbitrary<ZonedDateTime> configure(Arbitrary<ZonedDateTime> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof ZonedDateTimeArbitrary) {
				ZonedDateTimeArbitrary zonedDateTimeArbitrary = (ZonedDateTimeArbitrary) arbitrary;
				return zonedDateTimeArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForLocalTime extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(LocalTime.class);
		}

		public Arbitrary<LocalTime> configure(Arbitrary<LocalTime> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof LocalTimeArbitrary) {
				LocalTimeArbitrary localTimeArbitrary = (LocalTimeArbitrary) arbitrary;
				return localTimeArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForOffsetTime extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(OffsetTime.class);
		}

		public Arbitrary<OffsetTime> configure(Arbitrary<OffsetTime> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof OffsetTimeArbitrary) {
				OffsetTimeArbitrary offsetTimeArbitrary = (OffsetTimeArbitrary) arbitrary;
				return offsetTimeArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	public static class ForDuration extends ArbitraryConfiguratorBase {

		@Override
		protected boolean acceptTargetType(TypeUsage targetType) {
			return targetType.isAssignableFrom(Duration.class);
		}

		public Arbitrary<Duration> configure(Arbitrary<Duration> arbitrary, Precision range) {
			ChronoUnit ofPrecision = range.value();
			if (arbitrary instanceof DurationArbitrary) {
				DurationArbitrary durationArbitrary = (DurationArbitrary) arbitrary;
				return durationArbitrary.ofPrecision(ofPrecision);
			} else {
				return arbitrary.filter(v -> filter(v, ofPrecision));
			}
		}

	}

	private static boolean filter(int minute, int second, int nano, ChronoUnit ofPrecision) {
		switch (ofPrecision) {
			case HOURS:
				if (minute != 0) return false;
			case MINUTES:
				if (second != 0) return false;
			case SECONDS:
				if (nano != 0) return false;
				break;
			case MILLIS:
				if (nano % 1_000_000 != 0) return false;
				break;
			case MICROS:
				if (nano % 1_000 != 0) return false;
		}
		return true;
	}

	private static boolean filter(LocalDateTime dateTime, ChronoUnit ofPrecision) {
		return filter(dateTime.toLocalTime(), ofPrecision);
	}

	private static boolean filter(Instant instant, ChronoUnit ofPrecision) {
		if (LocalDateTime.MIN.toInstant(ZoneOffset.UTC).isAfter(instant) || LocalDateTime.MAX.toInstant(ZoneOffset.UTC).isBefore(instant)) {
			return false;
		}
		return filter(DefaultInstantArbitrary.instantToLocalDateTime(instant).toLocalTime(), ofPrecision);
	}

	private static boolean filter(OffsetDateTime dateTime, ChronoUnit ofPrecision) {
		return filter(dateTime.toLocalTime(), ofPrecision);
	}

	private static boolean filter(ZonedDateTime dateTime, ChronoUnit ofPrecision) {
		return filter(dateTime.toLocalTime(), ofPrecision);
	}

	private static boolean filter(LocalTime time, ChronoUnit ofPrecision) {
		return filter(time.getMinute(), time.getSecond(), time.getNano(), ofPrecision);
	}

	private static boolean filter(OffsetTime offsetTime, ChronoUnit ofPrecision) {
		LocalTime time = offsetTime.toLocalTime();
		return filter(time, ofPrecision);
	}

	private static boolean filter(Duration duration, ChronoUnit ofPrecision) {
		int minutes = (int) ((duration.getSeconds() % 3600) / 60);
		int seconds = (int) (duration.getSeconds() % 60);
		int nanos = duration.getNano();
		return filter(minutes, seconds, nanos, ofPrecision);
	}

}
