/*
 * Decompiled with CFR 0.152.
 */
package net.javacrumbs.shedlock.provider.s3v2;

import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import net.javacrumbs.shedlock.core.ClockProvider;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.provider.s3v2.Lock;
import net.javacrumbs.shedlock.support.AbstractStorageAccessor;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;

class S3StorageAccessor
extends AbstractStorageAccessor {
    private static final String LOCK_UNTIL = "lock-until";
    private static final String LOCKED_AT = "locked-at";
    private static final String LOCKED_BY = "locked-by";
    private static final int PRECONDITION_FAILED = 412;
    private final S3Client s3Client;
    private final String bucketName;
    private final String objectPrefix;

    public S3StorageAccessor(S3Client s3Client, String bucketName, String objectPrefix) {
        this.s3Client = s3Client;
        this.bucketName = bucketName;
        this.objectPrefix = objectPrefix;
    }

    Optional<Lock> find(String name, String action) {
        try {
            HeadObjectResponse metadataResponse = this.getExistingMetadata(name);
            Map metadata = metadataResponse.metadata();
            Instant lockUntil = Instant.parse((CharSequence)metadata.get(LOCK_UNTIL));
            Instant lockedAt = Instant.parse((CharSequence)metadata.get(LOCKED_AT));
            String lockedBy = (String)metadata.get(LOCKED_BY);
            String eTag = metadataResponse.eTag();
            this.logger.debug("Lock found. action: {}, name: {}, lockUntil: {}, e-tag: {}", new Object[]{action, name, lockUntil, eTag});
            return Optional.of(new Lock(lockUntil, lockedAt, lockedBy, eTag));
        }
        catch (AwsServiceException e) {
            if (e.statusCode() == 404) {
                this.logger.debug("Lock not found. action: {}, name: {}", (Object)action, (Object)name);
                return Optional.empty();
            }
            throw e;
        }
    }

    public boolean insertRecord(LockConfiguration lockConfiguration) {
        String name = lockConfiguration.getName();
        if (this.find(name, "insertRecord").isPresent()) {
            this.logger.debug("Lock already exists. name: {}", (Object)name);
            return false;
        }
        try {
            Map<String, String> metadata = this.createMetadata(lockConfiguration.getLockAtMostUntil(), ClockProvider.now(), this.getHostname());
            PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucketName).key(this.objectName(name)).metadata(metadata).ifNoneMatch("*").build();
            this.s3Client.putObject(request, S3StorageAccessor.getLockContent());
            this.logger.debug("Lock created successfully. name: {}, metadata: {}", (Object)name, metadata);
            return true;
        }
        catch (AwsServiceException e) {
            if (e.statusCode() == 412) {
                this.logger.debug("Lock already in use. name: {}", (Object)name);
            } else {
                this.logger.warn("Failed to create lock. name: {}", (Object)name, (Object)e);
            }
            return false;
        }
    }

    public boolean updateRecord(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "updateRecord");
        if (lock.isEmpty()) {
            this.logger.warn("Update skipped. Lock not found. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        if (lock.get().lockUntil().isAfter(ClockProvider.now())) {
            this.logger.debug("Update skipped. Lock still valid. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        Map<String, String> newMetadata = this.createMetadata(lockConfiguration.getLockAtMostUntil(), ClockProvider.now(), this.getHostname());
        return this.replaceObjectMetadata(lockConfiguration.getName(), newMetadata, lock.get().eTag(), "updateRecord");
    }

    public void unlock(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "unlock");
        if (lock.isEmpty()) {
            this.logger.warn("Unlock skipped. Lock not found. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return;
        }
        this.updateUntil(lockConfiguration.getName(), lock.get(), lockConfiguration.getUnlockTime(), "unlock");
    }

    public boolean extend(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "extend");
        if (lock.isEmpty() || lock.get().lockUntil().isBefore(ClockProvider.now()) || !lock.get().lockedBy().equals(this.getHostname())) {
            this.logger.debug("Extend skipped. Lock invalid or not owned by host. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        return this.updateUntil(lockConfiguration.getName(), lock.get(), lockConfiguration.getLockAtMostUntil(), "extend");
    }

    private boolean updateUntil(String name, Lock lock, Instant until, String action) {
        HeadObjectResponse existingMetadata = this.getExistingMetadata(name);
        Map<String, String> newMetadata = this.createMetadata(until, Instant.parse((CharSequence)existingMetadata.metadata().get(LOCKED_AT)), this.getHostname());
        return this.replaceObjectMetadata(name, newMetadata, lock.eTag(), action);
    }

    private HeadObjectResponse getExistingMetadata(String name) {
        return this.s3Client.headObject((HeadObjectRequest)HeadObjectRequest.builder().bucket(this.bucketName).key(this.objectName(name)).build());
    }

    private boolean replaceObjectMetadata(String name, Map<String, String> newMetadata, String eTag, String action) {
        PutObjectRequest request = (PutObjectRequest)PutObjectRequest.builder().bucket(this.bucketName).key(this.objectName(name)).metadata(newMetadata).ifMatch(eTag).build();
        try {
            PutObjectResponse response = this.s3Client.putObject(request, S3StorageAccessor.getLockContent());
            this.logger.debug("Lock {} successfully. name: {}, old e-tag: {}, new e-tag: {}", new Object[]{action, name, eTag, response.eTag()});
            return true;
        }
        catch (AwsServiceException e) {
            if (e.statusCode() == 412) {
                this.logger.debug("Lock not exists to {}. name: {}, e-tag {}", new Object[]{action, name, eTag});
            } else {
                this.logger.warn("Failed to {} lock. name: {}", new Object[]{action, name, e});
            }
            return false;
        }
    }

    private static RequestBody getLockContent() {
        UUID uuid = UUID.randomUUID();
        ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        return RequestBody.fromBytes((byte[])bb.array());
    }

    private Map<String, String> createMetadata(Instant lockUntil, Instant lockedAt, String lockedBy) {
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put(LOCK_UNTIL, lockUntil.toString());
        metadata.put(LOCKED_AT, lockedAt.toString());
        metadata.put(LOCKED_BY, lockedBy);
        return metadata;
    }

    private String objectName(String name) {
        return this.objectPrefix + name;
    }
}

