/*
 * Decompiled with CFR 0.152.
 */
package net.javacrumbs.shedlock.provider.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.Optional;
import java.util.UUID;
import net.javacrumbs.shedlock.core.ClockProvider;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.provider.s3.Lock;
import net.javacrumbs.shedlock.support.AbstractStorageAccessor;

class S3StorageAccessor
extends AbstractStorageAccessor {
    private static final String LOCK_UNTIL = "lockUntil";
    private static final String LOCKED_AT = "lockedAt";
    private static final String LOCKED_BY = "lockedBy";
    private static final int PRECONDITION_FAILED = 412;
    private final AmazonS3 s3Client;
    private final String bucketName;
    private final String objectPrefix;

    public S3StorageAccessor(AmazonS3 s3Client, String bucketName, String objectPrefix) {
        this.s3Client = s3Client;
        this.bucketName = bucketName;
        this.objectPrefix = objectPrefix;
    }

    Optional<Lock> find(String name, String action) {
        try {
            ObjectMetadata metadata = this.getExistingMetadata(name);
            Instant lockUntil = Instant.parse(metadata.getUserMetaDataOf(LOCK_UNTIL));
            Instant lockedAt = Instant.parse(metadata.getUserMetaDataOf(LOCKED_AT));
            String lockedBy = metadata.getUserMetaDataOf(LOCKED_BY);
            String eTag = metadata.getETag();
            this.logger.debug("Lock found. action: {}, name: {}, lockUntil: {}, e-tag: {}", new Object[]{action, name, lockUntil, eTag});
            return Optional.of(new Lock(lockUntil, lockedAt, lockedBy, eTag));
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 404) {
                this.logger.debug("Lock not found. action: {}, name: {}", (Object)action, (Object)name);
                return Optional.empty();
            }
            throw e;
        }
    }

    public boolean insertRecord(LockConfiguration lockConfiguration) {
        String name = lockConfiguration.getName();
        if (this.find(name, "insertRecord").isPresent()) {
            this.logger.debug("Lock already exists. name: {}", (Object)name);
            return false;
        }
        try {
            byte[] lockContent = S3StorageAccessor.getLockContent();
            ObjectMetadata metadata = this.createMetadata(lockConfiguration.getLockAtMostUntil(), ClockProvider.now(), this.getHostname());
            metadata.setContentLength((long)lockContent.length);
            PutObjectRequest request = new PutObjectRequest(this.bucketName, this.objectName(name), (InputStream)new ByteArrayInputStream(lockContent), metadata);
            request.putCustomRequestHeader("If-None-Match", "*");
            this.s3Client.putObject(request);
            this.logger.debug("Lock created successfully. name: {}, metadata: {}", (Object)name, (Object)metadata.getUserMetadata());
            return true;
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 412) {
                this.logger.debug("Lock already in use. name: {}", (Object)name);
            } else {
                this.logger.warn("Failed to create lock. name: {}", (Object)name, (Object)e);
            }
            return false;
        }
    }

    public boolean updateRecord(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "updateRecord");
        if (lock.isEmpty()) {
            this.logger.warn("Update skipped. Lock not found. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        if (lock.get().lockUntil().isAfter(ClockProvider.now())) {
            this.logger.debug("Update skipped. Lock still valid. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        ObjectMetadata newMetadata = this.createMetadata(lockConfiguration.getLockAtMostUntil(), ClockProvider.now(), this.getHostname());
        return this.replaceObjectMetadata(lockConfiguration.getName(), newMetadata, lock.get().eTag(), "updateRecord");
    }

    public void unlock(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "unlock");
        if (lock.isEmpty()) {
            this.logger.warn("Unlock skipped. Lock not found. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return;
        }
        this.updateUntil(lockConfiguration.getName(), lock.get(), lockConfiguration.getUnlockTime(), "unlock");
    }

    public boolean extend(LockConfiguration lockConfiguration) {
        Optional<Lock> lock = this.find(lockConfiguration.getName(), "extend");
        if (lock.isEmpty() || lock.get().lockUntil().isBefore(ClockProvider.now()) || !lock.get().lockedBy().equals(this.getHostname())) {
            this.logger.debug("Extend skipped. Lock invalid or not owned by host. name: {}, lock: {}", (Object)lockConfiguration.getName(), lock);
            return false;
        }
        return this.updateUntil(lockConfiguration.getName(), lock.get(), lockConfiguration.getLockAtMostUntil(), "extend");
    }

    private boolean updateUntil(String name, Lock lock, Instant until, String action) {
        ObjectMetadata existingMetadata = this.getExistingMetadata(name);
        ObjectMetadata newMetadata = this.createMetadata(until, Instant.parse(existingMetadata.getUserMetaDataOf(LOCKED_AT)), this.getHostname());
        return this.replaceObjectMetadata(name, newMetadata, lock.eTag(), action);
    }

    private ObjectMetadata getExistingMetadata(String name) {
        return this.s3Client.getObjectMetadata(this.bucketName, this.objectName(name));
    }

    private boolean replaceObjectMetadata(String name, ObjectMetadata newMetadata, String eTag, String action) {
        byte[] lockContent = S3StorageAccessor.getLockContent();
        newMetadata.setContentLength((long)lockContent.length);
        PutObjectRequest request = new PutObjectRequest(this.bucketName, this.objectName(name), (InputStream)new ByteArrayInputStream(lockContent), newMetadata);
        request.putCustomRequestHeader("If-Match", eTag);
        try {
            PutObjectResult response = this.s3Client.putObject(request);
            this.logger.debug("Lock {} successfully. name: {}, old e-tag: {}, new e-tag: {}", new Object[]{action, name, eTag, response.getETag()});
            return true;
        }
        catch (AmazonServiceException e) {
            if (e.getStatusCode() == 412) {
                this.logger.debug("Lock not exists to {}. name: {}, e-tag {}", new Object[]{action, name, eTag});
            } else {
                this.logger.warn("Failed to {} lock. name: {}", new Object[]{action, name, e});
            }
            return false;
        }
    }

    private static byte[] getLockContent() {
        UUID uuid = UUID.randomUUID();
        ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        return bb.array();
    }

    private ObjectMetadata createMetadata(Instant lockUntil, Instant lockedAt, String lockedBy) {
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.addUserMetadata(LOCK_UNTIL, lockUntil.toString());
        metadata.addUserMetadata(LOCKED_AT, lockedAt.toString());
        metadata.addUserMetadata(LOCKED_BY, lockedBy);
        return metadata;
    }

    private String objectName(String name) {
        return this.objectPrefix + name;
    }
}

