/*
 * Decompiled with CFR 0.152.
 */
package net.javacrumbs.shedlock.provider.consul;

import com.ecwid.consul.v1.ConsulClient;
import com.ecwid.consul.v1.QueryParams;
import com.ecwid.consul.v1.kv.model.PutParams;
import com.ecwid.consul.v1.session.model.NewSession;
import com.ecwid.consul.v1.session.model.Session;
import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.javacrumbs.shedlock.core.ClockProvider;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.core.LockProvider;
import net.javacrumbs.shedlock.core.SimpleLock;
import net.javacrumbs.shedlock.provider.consul.ConsulSimpleLock;
import net.javacrumbs.shedlock.support.annotation.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulLockProvider
implements LockProvider,
AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ConsulLockProvider.class);
    private static final String DEFAULT_CONSUL_LOCK_POSTFIX = "-leader";
    private static final Duration DEFAULT_GRACEFUL_SHUTDOWN_INTERVAL = Duration.ofSeconds(2L);
    private static final Duration DEFAULT_MIN_SESSION_TTL = Duration.ofSeconds(10L);
    private final ScheduledExecutorService unlockScheduler = Executors.newSingleThreadScheduledExecutor();
    private final Configuration configuration;

    public ConsulLockProvider(@NonNull ConsulClient consulClient) {
        this(Configuration.builder().withConsulClient(consulClient).build());
    }

    public ConsulLockProvider(@NonNull ConsulClient consulClient, Duration minSessionTtl) {
        this(Configuration.builder().withConsulClient(consulClient).withMinSessionTtl(minSessionTtl).build());
    }

    public ConsulLockProvider(@NonNull ConsulClient consulClient, Duration minSessionTtl, String consulLockPostfix, Duration gracefulShutdownInterval) {
        this(Configuration.builder().withConsulClient(consulClient).withMinSessionTtl(minSessionTtl).withConsulLockPostfix(consulLockPostfix).withGracefulShutdownInterval(gracefulShutdownInterval).build());
    }

    public ConsulLockProvider(@NonNull Configuration configuration) {
        this.configuration = configuration;
    }

    @NonNull
    public Optional<SimpleLock> lock(@NonNull LockConfiguration lockConfiguration) {
        String sessionId = this.createSession(lockConfiguration);
        return this.tryLock(sessionId, lockConfiguration);
    }

    void unlock(String sessionId, LockConfiguration lockConfiguration) {
        Duration additionalSessionTtl = Duration.between(ClockProvider.now(), lockConfiguration.getLockAtLeastUntil());
        if (!additionalSessionTtl.isNegative() && !additionalSessionTtl.isZero()) {
            logger.debug("Lock will still be held for {}", (Object)additionalSessionTtl);
            this.scheduleUnlock(sessionId, additionalSessionTtl);
        } else {
            this.destroy(sessionId);
        }
    }

    private String createSession(LockConfiguration lockConfiguration) {
        long ttlInSeconds = Math.max(lockConfiguration.getLockAtMostFor().getSeconds(), this.configuration.getMinSessionTtl().getSeconds());
        NewSession newSession = new NewSession();
        newSession.setName(lockConfiguration.getName());
        newSession.setLockDelay(0L);
        newSession.setBehavior(Session.Behavior.DELETE);
        newSession.setTtl(ttlInSeconds + "s");
        String sessionId = (String)this.client().sessionCreate(newSession, QueryParams.DEFAULT, this.token()).getValue();
        logger.debug("Acquired session {} for {} seconds", (Object)sessionId, (Object)ttlInSeconds);
        return sessionId;
    }

    private Optional<SimpleLock> tryLock(String sessionId, LockConfiguration lockConfiguration) {
        PutParams putParams = new PutParams();
        putParams.setAcquireSession(sessionId);
        String leaderKey = this.getLeaderKey(lockConfiguration);
        boolean isLockSuccessful = (Boolean)this.client().setKVValue(leaderKey, lockConfiguration.getName(), this.token(), putParams).getValue();
        if (isLockSuccessful) {
            return Optional.of(new ConsulSimpleLock(lockConfiguration, this, sessionId));
        }
        this.destroy(sessionId);
        return Optional.empty();
    }

    private String token() {
        return this.configuration.getToken();
    }

    private String getLeaderKey(LockConfiguration lockConfiguration) {
        if (this.configuration.getConsulLockPrefix() == null) {
            return lockConfiguration.getName() + this.configuration.getConsulLockPostfix();
        }
        return this.configuration.getConsulLockPrefix() + "/" + lockConfiguration.getName() + this.configuration.getConsulLockPostfix();
    }

    private void scheduleUnlock(String sessionId, Duration unlockTime) {
        this.unlockScheduler.schedule(this.catchExceptions(() -> this.destroy(sessionId)), unlockTime.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void destroy(String sessionId) {
        logger.debug("Destroying session {}", (Object)sessionId);
        this.client().sessionDestroy(sessionId, QueryParams.DEFAULT, this.token());
    }

    private Runnable catchExceptions(Runnable runnable) {
        return () -> {
            try {
                runnable.run();
            }
            catch (Throwable t) {
                logger.warn("Exception while execution", t);
            }
        };
    }

    @Override
    public void close() {
        this.unlockScheduler.shutdown();
        try {
            if (!this.unlockScheduler.awaitTermination(this.configuration.getGracefulShutdownInterval().toMillis(), TimeUnit.MILLISECONDS)) {
                this.unlockScheduler.shutdownNow();
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private ConsulClient client() {
        return this.configuration.getConsulClient();
    }

    public static final class Configuration {
        private final Duration minSessionTtl;
        private final String consulLockPrefix;
        private final String consulLockPostfix;
        private final ConsulClient consulClient;
        private final Duration gracefulShutdownInterval;
        private final String token;

        Configuration(Duration minSessionTtl, String consulLockPostfix, ConsulClient consulClient, Duration gracefulShutdownInterval, String token, String consulLockPrefix) {
            this.minSessionTtl = minSessionTtl;
            this.consulLockPrefix = consulLockPrefix;
            this.consulLockPostfix = consulLockPostfix;
            this.consulClient = Objects.requireNonNull(consulClient, "consulClient can not be null");
            this.gracefulShutdownInterval = gracefulShutdownInterval;
            this.token = token;
        }

        public Duration getMinSessionTtl() {
            return this.minSessionTtl;
        }

        public String getConsulLockPostfix() {
            return this.consulLockPostfix;
        }

        public ConsulClient getConsulClient() {
            return this.consulClient;
        }

        public Duration getGracefulShutdownInterval() {
            return this.gracefulShutdownInterval;
        }

        public String getToken() {
            return this.token;
        }

        public String getConsulLockPrefix() {
            return this.consulLockPrefix;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static final class Builder {
            private Duration minSessionTtl = DEFAULT_MIN_SESSION_TTL;
            private String consulLockPostfix = "-leader";
            private ConsulClient consulClient;
            private Duration gracefulShutdownInterval = DEFAULT_GRACEFUL_SHUTDOWN_INTERVAL;
            private String token;
            private String consulLockPrefix;

            public Builder withMinSessionTtl(Duration minSessionTtl) {
                this.minSessionTtl = minSessionTtl;
                return this;
            }

            public Builder withConsulLockPostfix(String consulLockPostfix) {
                this.consulLockPostfix = consulLockPostfix;
                return this;
            }

            public Builder withConsulClient(ConsulClient consulClient) {
                this.consulClient = consulClient;
                return this;
            }

            public Builder withGracefulShutdownInterval(Duration gracefulShutdownInterval) {
                this.gracefulShutdownInterval = gracefulShutdownInterval;
                return this;
            }

            public Builder withToken(String token) {
                this.token = token;
                return this;
            }

            public Builder withConsulLockPrefix(String leaderKeyPrefix) {
                this.consulLockPrefix = leaderKeyPrefix;
                return this;
            }

            public Configuration build() {
                return new Configuration(this.minSessionTtl, this.consulLockPostfix, this.consulClient, this.gracefulShutdownInterval, this.token, this.consulLockPrefix);
            }
        }
    }
}

