/*
 * Decompiled with CFR 0.152.
 */
package net.javacrumbs.shedlock.provider.consul;

import com.ecwid.consul.v1.ConsulClient;
import com.ecwid.consul.v1.QueryParams;
import com.ecwid.consul.v1.kv.model.PutParams;
import com.ecwid.consul.v1.session.model.NewSession;
import com.ecwid.consul.v1.session.model.Session;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.javacrumbs.shedlock.core.ClockProvider;
import net.javacrumbs.shedlock.core.LockConfiguration;
import net.javacrumbs.shedlock.core.LockProvider;
import net.javacrumbs.shedlock.core.SimpleLock;
import net.javacrumbs.shedlock.provider.consul.ConsulSimpleLock;
import net.javacrumbs.shedlock.support.annotation.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulLockProvider
implements LockProvider,
AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ConsulLockProvider.class);
    private static final String DEFAULT_CONSUL_LOCK_POSTFIX = "-leader";
    private static final Duration DEFAULT_GRACEFUL_SHUTDOWN_INTERVAL = Duration.ofSeconds(2L);
    private final ScheduledExecutorService unlockScheduler = Executors.newSingleThreadScheduledExecutor();
    private final Duration minSessionTtl;
    private final String consulLockPostfix;
    private final ConsulClient consulClient;
    private final Duration gracefulShutdownInterval;

    public ConsulLockProvider(ConsulClient consulClient) {
        this(consulClient, Duration.ofSeconds(10L), DEFAULT_CONSUL_LOCK_POSTFIX, DEFAULT_GRACEFUL_SHUTDOWN_INTERVAL);
    }

    public ConsulLockProvider(ConsulClient consulClient, Duration minSessionTtl) {
        this(consulClient, minSessionTtl, DEFAULT_CONSUL_LOCK_POSTFIX, DEFAULT_GRACEFUL_SHUTDOWN_INTERVAL);
    }

    public ConsulLockProvider(ConsulClient consulClient, Duration minSessionTtl, String consulLockPostfix, Duration gracefulShutdownInterval) {
        this.consulClient = consulClient;
        this.consulLockPostfix = consulLockPostfix;
        this.minSessionTtl = minSessionTtl;
        this.gracefulShutdownInterval = gracefulShutdownInterval;
        Runtime.getRuntime().addShutdownHook(new Thread(this::close));
    }

    @NonNull
    public Optional<SimpleLock> lock(@NonNull LockConfiguration lockConfiguration) {
        String sessionId = this.createSession(lockConfiguration);
        return this.tryLock(sessionId, lockConfiguration);
    }

    void unlock(String sessionId, LockConfiguration lockConfiguration) {
        Duration additionalSessionTtl = Duration.between(ClockProvider.now(), lockConfiguration.getLockAtLeastUntil());
        if (!additionalSessionTtl.isNegative() && !additionalSessionTtl.isZero()) {
            logger.debug("Lock will still be held for {}", (Object)additionalSessionTtl);
            this.scheduleUnlock(sessionId, additionalSessionTtl);
        } else {
            this.destroy(sessionId);
        }
    }

    private String createSession(LockConfiguration lockConfiguration) {
        long ttlInSeconds = Math.max(lockConfiguration.getLockAtMostFor().getSeconds(), this.minSessionTtl.getSeconds());
        NewSession newSession = new NewSession();
        newSession.setName(lockConfiguration.getName());
        newSession.setLockDelay(0L);
        newSession.setBehavior(Session.Behavior.DELETE);
        newSession.setTtl(ttlInSeconds + "s");
        String sessionId = (String)this.consulClient.sessionCreate(newSession, QueryParams.DEFAULT).getValue();
        logger.debug("Acquired session {} for {} seconds", (Object)sessionId, (Object)ttlInSeconds);
        return sessionId;
    }

    private Optional<SimpleLock> tryLock(String sessionId, LockConfiguration lockConfiguration) {
        PutParams putParams = new PutParams();
        putParams.setAcquireSession(sessionId);
        String leaderKey = this.getLeaderKey(lockConfiguration);
        boolean isLockSuccessful = (Boolean)this.consulClient.setKVValue(leaderKey, lockConfiguration.getName(), putParams).getValue();
        if (isLockSuccessful) {
            return Optional.of(new ConsulSimpleLock(lockConfiguration, this, sessionId));
        }
        return Optional.empty();
    }

    private String getLeaderKey(LockConfiguration lockConfiguration) {
        return lockConfiguration.getName() + this.consulLockPostfix;
    }

    private void scheduleUnlock(String sessionId, Duration unlockTime) {
        this.unlockScheduler.schedule(this.catchExceptions(() -> this.destroy(sessionId)), unlockTime.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void destroy(String sessionId) {
        logger.debug("Destroying session {}", (Object)sessionId);
        this.consulClient.sessionDestroy(sessionId, QueryParams.DEFAULT);
    }

    private Runnable catchExceptions(Runnable runnable) {
        return () -> {
            try {
                runnable.run();
            }
            catch (Throwable t) {
                logger.warn("Exception while execution", t);
            }
        };
    }

    @Override
    public void close() {
        this.unlockScheduler.shutdown();
        try {
            if (!this.unlockScheduler.awaitTermination(this.gracefulShutdownInterval.toMillis(), TimeUnit.MILLISECONDS)) {
                this.unlockScheduler.shutdownNow();
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }
}

