/*
 * Copyright 2015-2020 Austin Keener, Michael Ritter, Florian Spieß, and the JDA contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.dv8tion.jda.api.entities;

import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.Permission;
import net.dv8tion.jda.api.exceptions.HttpException;
import net.dv8tion.jda.api.exceptions.InsufficientPermissionException;
import net.dv8tion.jda.api.requests.RestAction;
import net.dv8tion.jda.api.requests.restaction.AuditableRestAction;
import net.dv8tion.jda.api.requests.restaction.MessageAction;
import net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction;
import net.dv8tion.jda.internal.JDAImpl;
import net.dv8tion.jda.internal.requests.FunctionalCallback;
import net.dv8tion.jda.internal.requests.Requester;
import net.dv8tion.jda.internal.utils.Checks;
import net.dv8tion.jda.internal.utils.IOUtil;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import org.apache.commons.collections4.Bag;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.*;
import java.time.OffsetDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * Represents a Text message received from Discord.
 * <br>This represents messages received from {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannels}.
 *
 * <p><b>This type is not updated. JDA does not keep track of changes to messages, it is advised to do this via events such
 * as {@link net.dv8tion.jda.api.events.message.MessageUpdateEvent MessageUpdateEvent} and similar.</b>
 *
 * <h2>Message Differences</h2>
 * There are 3 implementations of this interface in JDA.
 * <ol>
 *     <li><b>Received Message</b>
 *     <br>Messages received through events or history query.
 *         These messages hold information of <b>existing</b> messages and
 *         can be modified or deleted.</li>
 *     <li><b>System Message</b>
 *     <br>Specification of <b>Received Messages</b> that are generated by Discord
 *         on certain events. Commonly this is used in groups or to indicate a pin within a MessageChannel.
 *         The different types can be found in the {@link net.dv8tion.jda.api.entities.MessageType MessageType} enum.</li>
 *     <li><b>Data Message</b>
 *     <br>This type is produced by {@link net.dv8tion.jda.api.MessageBuilder MessageBuilder}
 *         and only holds sendable information such as content or nonce. These messages do not allow
 *         any modifications via RestActions or information that is generated when sent such as the id to be used.</li>
 * </ol>
 *
 * <p>When a feature is not available it will throw an {@link java.lang.UnsupportedOperationException UnsupportedOperationException}
 * as per interface specifications.
 * <br>Specific operations may have specified information available in the {@code throws} javadoc.
 *
 * <h1>Formattable</h1>
 * This interface extends {@link java.util.Formattable Formattable} and can be used with a {@link java.util.Formatter Formatter}
 * such as used by {@link String#format(String, Object...) String.format(String, Object...)}
 * or {@link java.io.PrintStream#printf(String, Object...) PrintStream.printf(String, Object...)}.
 *
 * <p>This will use {@link #getContentDisplay()} rather than {@link Object#toString()}!
 * <br>Supported Features:
 * <ul>
 *     <li><b>Alternative</b>
 *     <br>   - Using {@link #getContentRaw()}
 *              (Example: {@code %#s} - uses {@link #getContentDisplay()})</li>
 *
 *     <li><b>Width/Left-Justification</b>
 *     <br>   - Ensures the size of a format
 *              (Example: {@code %20s} - uses at minimum 20 chars;
 *              {@code %-10s} - uses left-justified padding)</li>
 *
 *     <li><b>Precision</b>
 *     <br>   - Cuts the content to the specified size
 *              (replacing last 3 chars with {@code ...}; Example: {@code %.20s})</li>
 * </ul>
 *
 * <p>More information on formatting syntax can be found in the {@link java.util.Formatter format syntax documentation}!
 *
 * @see net.dv8tion.jda.api.MessageBuilder MessageBuilder
 * @see MessageChannel#sendMessage(Message)
 *
 * @see MessageChannel#getIterableHistory()
 * @see MessageChannel#getHistory()
 * @see MessageChannel#getHistoryAfter(String, int)
 * @see MessageChannel#getHistoryBefore(String, int)
 * @see MessageChannel#getHistoryAround(String, int)
 * @see MessageChannel#getHistoryFromBeginning(int)
 * @see MessageChannel#retrieveMessageById(String)
 *
 * @see MessageChannel#deleteMessageById(String)
 * @see MessageChannel#editMessageById(String, CharSequence)
 */
public interface Message extends ISnowflake, Formattable
{
    /**
     * The maximum sendable file size (8 MiB)
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE = 8 << 20;

    /**
     * The maximum sendable file size for nitro (50 MiB)
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE_NITRO = 50 << 20;

    /**
     * The maximum amount of files sendable within a single message ({@value})
     *
     * @see MessageAction#addFile(java.io.File, net.dv8tion.jda.api.utils.AttachmentOption...) MessageAction.addFile(...)
     */
    int MAX_FILE_AMOUNT = 10;

    /**
     * The maximum amount of characters sendable in one message. ({@value})
     * <br>This only applies to the raw content and not embeds!
     *
     * @see MessageAction#append(CharSequence) MessageAction.append(...)
     */
    int MAX_CONTENT_LENGTH = 2000;

    /**
     * Pattern used to find instant invites in messages.
     *
     * @see #getInvites()
     */
    Pattern INVITE_PATTERN = Pattern.compile("(?:https?://)?discord(?:app\\.com/invite|\\.gg)/([a-z0-9-]+)", Pattern.CASE_INSENSITIVE);

    /**
     * An immutable list of all mentioned {@link net.dv8tion.jda.api.entities.User Users}.
     * <br>If no user was mentioned, this list is empty. Elements are sorted in order of appearance. This only
     * counts direct mentions of the user and not mentions through roles or the everyone tag.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned users
     */
    @Nonnull
    List<User> getMentionedUsers();

    /**
     * A {@link org.apache.commons.collections4.Bag Bag} of mentioned users.
     * <br>This can be used to retrieve the amount of times a user was mentioned in this message. This only
     * counts direct mentions of the user and not mentions through roles or the everyone tag.
     *
     * <h2>Example</h2>
     * <pre>{@code
     * public void sendCount(Message msg)
     * {
     *     List<User> mentions = msg.getMentionedUsers(); // distinct list, in order of appearance
     *     Bag<User> count = msg.getMentionedUsersBag();
     *     StringBuilder content = new StringBuilder();
     *     for (User user : mentions)
     *     {
     *         content.append(user.getAsTag())
     *                .append(": ")
     *                .append(count.getCount(user))
     *                .append("\n");
     *     }
     *     msg.getChannel().sendMessage(content.toString()).queue();
     * }
     * }</pre>
     *
     * @return {@link org.apache.commons.collections4.Bag Bag} of mentioned users
     *
     * @see    #getMentionedUsers()
     */
    @Nonnull
    Bag<User> getMentionedUsersBag();

    /**
     * A immutable list of all mentioned {@link net.dv8tion.jda.api.entities.TextChannel TextChannels}.
     * <br>If none were mentioned, this list is empty. Elements are sorted in order of appearance.
     *
     * <p><b>This may include TextChannels from other {@link net.dv8tion.jda.api.entities.Guild Guilds}</b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned TextChannels
     */
    @Nonnull
    List<TextChannel> getMentionedChannels();

    /**
     * A {@link org.apache.commons.collections4.Bag Bag} of mentioned channels.
     * <br>This can be used to retrieve the amount of times a channel was mentioned in this message.
     *
     * <h2>Example</h2>
     * <pre>{@code
     * public void sendCount(Message msg)
     * {
     *     List<TextChannel> mentions = msg.getMentionedTextChannels(); // distinct list, in order of appearance
     *     Bag<TextChannel> count = msg.getMentionedTextChannelsBag();
     *     StringBuilder content = new StringBuilder();
     *     for (TextChannel channel : mentions)
     *     {
     *         content.append("#")
     *                .append(channel.getName())
     *                .append(": ")
     *                .append(count.getCount(channel))
     *                .append("\n");
     *     }
     *     msg.getChannel().sendMessage(content.toString()).queue();
     * }
     * }</pre>
     *
     * @return {@link org.apache.commons.collections4.Bag Bag} of mentioned channels
     *
     * @see    #getMentionedChannels()
     */
    @Nonnull
    Bag<TextChannel> getMentionedChannelsBag();

    /**
     * A immutable list of all mentioned {@link net.dv8tion.jda.api.entities.Role Roles}.
     * <br>If none were mentioned, this list is empty. Elements are sorted in order of appearance. This only
     * counts direct mentions of the role and not mentions through the everyone tag.
     *
     * <p><b>This may include Roles from other {@link net.dv8tion.jda.api.entities.Guild Guilds}</b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned Roles
     */
    @Nonnull
    List<Role> getMentionedRoles();

    /**
     * A {@link org.apache.commons.collections4.Bag Bag} of mentioned roles.
     * <br>This can be used to retrieve the amount of times a role was mentioned in this message. This only
     * counts direct mentions of the role and not mentions through the everyone tag.
     * If a role is not {@link net.dv8tion.jda.api.entities.Role#isMentionable() mentionable} it will not be included.
     *
     * <h2>Example</h2>
     * <pre>{@code
     * public void sendCount(Message msg)
     * {
     *     List<Role> mentions = msg.getMentionedRoles(); // distinct list, in order of appearance
     *     Bag<Role> count = msg.getMentionedRolesBag();
     *     StringBuilder content = new StringBuilder();
     *     for (Role role : mentions)
     *     {
     *         content.append(role.getName())
     *                .append(": ")
     *                .append(count.getCount(role))
     *                .append("\n");
     *     }
     *     msg.getChannel().sendMessage(content.toString()).queue();
     * }
     * }</pre>
     *
     * @return {@link org.apache.commons.collections4.Bag Bag} of mentioned roles
     *
     * @see    #getMentionedRoles()
     */
    @Nonnull
    Bag<Role> getMentionedRolesBag();

    /**
     * Creates an immutable list of {@link net.dv8tion.jda.api.entities.Member Members}
     * representing the users of {@link #getMentionedUsers()} in the specified
     * {@link net.dv8tion.jda.api.entities.Guild Guild}.
     * <br>This is only a convenience method and will skip all users that are not in the specified
     * Guild.
     *
     * @param  guild
     *         Non-null {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         that will be used to retrieve Members.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalArgumentException
     *         If the specified Guild is {@code null}
     *
     * @return Immutable list of mentioned Members
     *
     * @since  3.4.0
     */
    @Nonnull
    List<Member> getMentionedMembers(@Nonnull Guild guild);

    /**
     * Creates an immutable list of {@link net.dv8tion.jda.api.entities.Member Members}
     * representing the users of {@link #getMentionedUsers()} in the
     * {@link net.dv8tion.jda.api.entities.Guild Guild} this Message was sent in.
     * <br>This is only a convenience method and will skip all users that are not in the specified Guild.
     * <br>It will provide the {@link #getGuild()} output Guild to {@link #getMentionedMembers(Guild)}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If this message was not sent in a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *
     * @return Immutable list of mentioned Members
     *
     * @since  3.4.0
     */
    @Nonnull
    List<Member> getMentionedMembers();

    /**
     * Combines all instances of {@link net.dv8tion.jda.api.entities.IMentionable IMentionable}
     * filtered by the specified {@link net.dv8tion.jda.api.entities.Message.MentionType MentionType} values.
     * <br>This does not include {@link #getMentionedMembers()} to avoid duplicates.
     *
     * <p>If no MentionType values are given this will fallback to all types.
     *
     * @param  types
     *         Amount of {@link net.dv8tion.jda.api.entities.Message.MentionType MentionTypes}
     *         to include in the list of mentions
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalArgumentException
     *         If provided with {@code null}
     *
     * @return Immutable list of filtered {@link net.dv8tion.jda.api.entities.IMentionable IMentionable} instances
     *
     * @since  3.4.0
     */
    @Nonnull
    List<IMentionable> getMentions(@Nonnull MentionType... types);

    /**
     * Checks if given {@link net.dv8tion.jda.api.entities.IMentionable IMentionable}
     * was mentioned in this message in any way (@User, @everyone, @here, @Role).
     * <br>If no filtering {@link net.dv8tion.jda.api.entities.Message.MentionType MentionTypes} are
     * specified this will fallback to all mention types.
     *
     * <p>{@link Message.MentionType#HERE MentionType.HERE} and {@link Message.MentionType#EVERYONE MentionType.EVERYONE}
     * will only be checked, if the given {@link net.dv8tion.jda.api.entities.IMentionable IMentionable} is of type
     * {@link net.dv8tion.jda.api.entities.User User} or {@link net.dv8tion.jda.api.entities.Member Member}.
     * <br>Online status of Users/Members is <b>NOT</b> considered when checking {@link Message.MentionType#HERE MentionType.HERE}.
     *
     * @param  mentionable
     *         The mentionable entity to check on.
     * @param  types
     *         The types to include when checking whether this type was mentioned.
     *         This will be used with {@link #getMentions(Message.MentionType...) getMentions(MentionType...)}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return True, if the given mentionable was mentioned in this message
     */
    boolean isMentioned(@Nonnull IMentionable mentionable, @Nonnull MentionType... types);

    /**
     * Indicates if this Message mentions everyone using @everyone or @here.
     *
     * @return True, if message is mentioning everyone
     */
    boolean mentionsEveryone();

    /**
     * Returns whether or not this Message has been edited before.
     *
     * @return True if this message has been edited.
     */
    boolean isEdited();

    /**
     * Provides the {@link java.time.OffsetDateTime OffsetDateTime} defining when this Message was last
     * edited. If this Message has not been edited ({@link #isEdited()} is {@code false}), then this method
     * will return {@code null}.
     *
     * @return Time of the most recent edit, or {@code null} if the Message has never been edited.
     */
    @Nullable
    OffsetDateTime getTimeEdited();

    /**
     * The author of this Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return Message author
     */
    @Nonnull
    User getAuthor();

    /**
     * Returns the author of this Message as a {@link net.dv8tion.jda.api.entities.Member member}.
     * <br>This is just a shortcut to {@link #getGuild()}{@link net.dv8tion.jda.api.entities.Guild#getMember(User) .getMember(getAuthor())}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b> This will return {@code null}
     * if it was not sent from a TextChannel.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return Message author, or {@code null} if the message was not sent from a TextChannel.
     */
    @Nullable
    Member getMember();

    /**
     * Returns the jump-to URL for the received message. Clicking this URL in the Discord client will cause the client to
     * jump to the specified message.
     * 
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * 
     * @return A String representing the jump-to URL for the message
     */
    @Nonnull
    String getJumpUrl();

    /**
     * The textual content of this message in the format that would be shown to the Discord client. All
     * {@link net.dv8tion.jda.api.entities.IMentionable IMentionable} entities will be resolved to the format
     * shown by the Discord client instead of the {@literal <id>} format.
     *
     * <p>This includes resolving:
     * <br>{@link net.dv8tion.jda.api.entities.User Users} / {@link net.dv8tion.jda.api.entities.Member Members}
     * to their @Username/@Nickname format,
     * <br>{@link net.dv8tion.jda.api.entities.TextChannel TextChannels} to their #ChannelName format,
     * <br>{@link net.dv8tion.jda.api.entities.Role Roles} to their @RoleName format
     * <br>{@link net.dv8tion.jda.api.entities.Emote Emotes} (not emojis!) to their {@code :name:} format.
     *
     * <p>If you want the actual Content (mentions as {@literal <@id>}), use {@link #getContentRaw()} instead
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The textual content of the message with mentions resolved to be visually like the Discord client.
     */
    @Nonnull
    String getContentDisplay();

    /**
     * The raw textual content of this message. Does not resolve {@link net.dv8tion.jda.api.entities.IMentionable IMentionable}
     * entities like {@link #getContentDisplay()} does. This means that this is the completely raw textual content of the message
     * received from Discord and can contain mentions specified by
     * <a href="https://discordapp.com/developers/docs/resources/channel#message-formatting" target="_blank">Discord's Message Formatting</a>.
     *
     * @return The raw textual content of the message, containing unresolved Discord message formatting.
     */
    @Nonnull
    String getContentRaw();

    /**
     * Gets the textual content of this message using {@link #getContentDisplay()} and then strips it of markdown characters 
     * like {@literal *, **, __, ~~, ||} that provide text formatting. Any characters that match these but are not being used 
     * for formatting are escaped to prevent possible formatting.
     *
     * @return The textual content from {@link #getContentDisplay()} with all text formatting characters removed or escaped.
     */
    @Nonnull
    String getContentStripped();

    /**
     * Creates an immutable List of {@link net.dv8tion.jda.api.entities.Invite Invite} codes
     * that are included in this Message.
     * <br>This will use the {@link java.util.regex.Pattern Pattern} provided
     * under {@link #INVITE_PATTERN} to construct a {@link java.util.regex.Matcher Matcher} that will
     * parse the {@link #getContentRaw()} output and include all codes it finds in a list.
     *
     * <p>You can use the codes to retrieve/validate invites via
     * {@link net.dv8tion.jda.api.entities.Invite#resolve(JDA, String) Invite.resolve(JDA, String)}
     *
     * @return Immutable list of invite codes
     */
    @Nonnull
    List<String> getInvites();

    /**
     * Validation <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank" >nonce</a> for this Message
     * <br>This can be used to validate that a Message was properly sent to the Discord Service.
     * <br>To set a nonce before sending you may use {@link net.dv8tion.jda.api.MessageBuilder#setNonce(String) MessageBuilder.setNonce(String)}!
     *
     * @return The validation nonce
     *
     * @since  3.4.0
     *
     * @see    net.dv8tion.jda.api.MessageBuilder#setNonce(String)
     * @see    <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank">Cryptographic Nonce - Wikipedia</a>
     */
    @Nullable
    String getNonce();

    /**
     * Used to determine if this Message was received from a {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * of the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} specified.
     * <br>This will always be false for {@link net.dv8tion.jda.api.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.api.entities.VoiceChannel VoiceChannels}.
     *
     * <p>Useful for restricting functionality to a certain type of channels.
     *
     * @param  type
     *         The {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} to check against.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return True if the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} which this message was received
     *         from is the same as the one specified by {@code type}.
     */
    boolean isFromType(@Nonnull ChannelType type);

    /**
     * Whether this message was sent in a {@link net.dv8tion.jda.api.entities.Guild Guild}.
     * <br>If this is {@code false} then {@link #getGuild()} will throw an {@link java.lang.IllegalStateException}.
     *
     * @return True, if {@link #getChannelType()}.{@link ChannelType#isGuild() isGuild()} is true.
     */
    default boolean isFromGuild()
    {
        return getChannelType().isGuild();
    }

    /**
     * Gets the {@link net.dv8tion.jda.api.entities.ChannelType ChannelType} that this message was received from.
     * <br>This will never be {@link net.dv8tion.jda.api.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.api.entities.VoiceChannel VoiceChannels}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The ChannelType which this message was received from.
     */
    @Nonnull
    ChannelType getChannelType();

    /**
     * Indicates if this Message was sent by a {@link net.dv8tion.jda.api.entities.Webhook Webhook} instead of a
     * {@link net.dv8tion.jda.api.entities.User User}.
     * <br>Useful if you want to ignore non-users.
     *
     * @return True if this message was sent by a {@link net.dv8tion.jda.api.entities.Webhook Webhook}.
     */
    boolean isWebhookMessage();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel} that this message was sent in.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The MessageChannel of this Message
     */
    @Nonnull
    MessageChannel getChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a PrivateChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.PrivateChannel}.
     *
     * @return The PrivateChannel this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    PrivateChannel getPrivateChannel();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @return The TextChannel this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    TextChannel getTextChannel();

    /**
     * The {@link net.dv8tion.jda.api.entities.Category Category} this
     * message was sent in. This will always be {@code null} for DMs.
     * <br>Equivalent to {@code getTextChannel().getParent()} if this was sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return {@link net.dv8tion.jda.api.entities.Category Category} for this message
     */
    @Nullable
    Category getCategory();

    /**
     * Returns the {@link net.dv8tion.jda.api.entities.Guild Guild} that this message was sent in.
     * <br>This is just a shortcut to {@link #getTextChannel()}.{@link net.dv8tion.jda.api.entities.TextChannel#getGuild() getGuild()}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b>
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If this was not sent in a {@link net.dv8tion.jda.api.entities.TextChannel}.
     *
     * @return The Guild this message was sent in
     *
     * @see    #isFromGuild()
     * @see    #isFromType(ChannelType)
     * @see    #getChannelType()
     */
    @Nonnull
    Guild getGuild();

    /**
     * An immutable list of {@link net.dv8tion.jda.api.entities.Message.Attachment Attachments} that are attached to this message.
     * <br>Most likely this will only ever be 1 {@link net.dv8tion.jda.api.entities.Message.Attachment Attachment} at most.
     *
     * @return Immutable list of {@link net.dv8tion.jda.api.entities.Message.Attachment Attachments}.
     */
    @Nonnull
    List<Attachment> getAttachments();

    /**
     * An immutable list of {@link net.dv8tion.jda.api.entities.MessageEmbed MessageEmbeds} that are part of this
     * Message.
     *
     * @return Immutable list of all given MessageEmbeds.
     */
    @Nonnull
    List<MessageEmbed> getEmbeds();

    /**
     * All {@link net.dv8tion.jda.api.entities.Emote Emotes} used in this Message.
     * <br><b>This only includes Custom Emotes, not unicode Emojis.</b> JDA classifies Emotes as the Custom Emojis uploaded
     * to a Guild and retrievable with {@link net.dv8tion.jda.api.entities.Guild#getEmotes()}. These are not the same
     * as the unicode emojis that Discord also supports. Elements are sorted in order of appearance.
     * <p>
     * <b>This may or may not contain fake Emotes which means they can be displayed but not used by the logged in account.</b>
     * To check whether an Emote is fake you can test if {@link Emote#isFake()} returns true.
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.api.entities.Emote Emote}!</u></b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return An immutable list of the Emotes used in this message (example match {@literal <:jda:230988580904763393>})
     */
    @Nonnull
    List<Emote> getEmotes();

    /**
     * A {@link org.apache.commons.collections4.Bag Bag} of emotes used in this message.
     * <br>This can be used to retrieve the amount of times an emote was used in this message.
     *
     * <h2>Example</h2>
     * <pre>{@code
     * public void sendCount(Message msg)
     * {
     *     List<Emote> emotes = msg.getEmotes(); // distinct list, in order of appearance
     *     Bag<Emote> count = msg.getEmotesBag();
     *     StringBuilder content = new StringBuilder();
     *     for (Emote emote : emotes)
     *     {
     *         content.append(emote.getName())
     *                .append(": ")
     *                .append(count.getCount(role))
     *                .append("\n");
     *     }
     *     msg.getChannel().sendMessage(content.toString()).queue();
     * }
     * }</pre>
     *
     * @return {@link org.apache.commons.collections4.Bag Bag} of used emotes
     *
     * @see    #getEmotes()
     */
    @Nonnull
    Bag<Emote> getEmotesBag();

    /**
     * All {@link net.dv8tion.jda.api.entities.MessageReaction MessageReactions} that are on this Message.
     *
     * @return Immutable list of all MessageReactions on this message.
     */
    @Nonnull
    List<MessageReaction> getReactions();

    /**
     * Defines whether or not this Message triggers TTS (Text-To-Speech).
     *
     * @return If this message is TTS.
     */
    boolean isTTS();

    /**
     * A {@link net.dv8tion.jda.api.entities.MessageActivity MessageActivity} that contains its type and party id.
     *
     * @return The activity, or {@code null} if no activity was added to the message.
     */
    @Nullable
    MessageActivity getActivity();

    /**
     * Edits this Message's content to the provided String.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account, or if
     *         {@code newContent}'s length is 0 or greater than 2000.
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessage(@Nonnull CharSequence newContent);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.api.entities.MessageEmbed MessageEmbed}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account, or
     *         if the passed-in embed is {@code null}
     *         or not {@link net.dv8tion.jda.api.entities.MessageEmbed#isSendable(net.dv8tion.jda.api.AccountType) sendable}
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessage(@Nonnull MessageEmbed newContent);

    /**
     * Edits this Message's content to the provided format.
     * <br>Shortcut for {@link net.dv8tion.jda.api.MessageBuilder#appendFormat(String, Object...)}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  format
     *         Format String used to generate the Message's content via
     *         {@link net.dv8tion.jda.api.MessageBuilder#appendFormat(String, Object...)} specification
     * @param  args
     *         The arguments to use in order to be converted in the format string
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws IllegalArgumentException
     *         If the provided format String is {@code null} or blank, or if
     *         the created message exceeds the 2000 character limit
     * @throws java.util.IllegalFormatException
     *         If a format string contains an illegal syntax,
     *         a format specifier that is incompatible with the given arguments,
     *         insufficient arguments given the format string, or other illegal conditions.
     *         For specification of all possible formatting errors,
     *         see the <a href="../util/Formatter.html#detail">Details</a>
     *         section of the formatter class specification.
     * @throws IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessageFormat(@Nonnull String format, @Nonnull Object... args);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.api.entities.Message Message}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If the message attempting to be edited was not created by the currently logged in account</li>
     *             <li>If the message contains a MessageEmbed that is not
     *                 {@link net.dv8tion.jda.api.entities.MessageEmbed#isSendable(net.dv8tion.jda.api.AccountType) sendable}</li>
     *         </ul>
     *
     * @return {@link MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.api.entities.Message Message} with the updated content
     */
    @Nonnull
    @CheckReturnValue
    MessageAction editMessage(@Nonnull Message newContent);

    /**
     * Deletes this Message from Discord.
     * <br>If this Message was not sent by the currently logged in account, then this will fail unless the Message is from
     * a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the current account has
     * {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     *
     * <p><u>To delete many messages at once in a {@link net.dv8tion.jda.api.entities.MessageChannel MessageChannel}
     * you should use {@link net.dv8tion.jda.api.entities.MessageChannel#purgeMessages(List) MessageChannel.purgeMessages(List)} instead.</u>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The delete was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The delete was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when deleting another Member's message
     *         or lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The message was already deleted at the time the request was sent.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link net.dv8tion.jda.api.MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message was not sent by the currently logged in account, the Message was sent in a
     *         {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}, and the currently logged in account
     *         does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the channel.
     * @throws java.lang.IllegalStateException
     *         If this Message was not sent by the currently logged in account and it was <b>not</b> sent in a
     *         {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.
     *
     * @return {@link net.dv8tion.jda.api.requests.restaction.AuditableRestAction AuditableRestAction}
     *
     * @see    net.dv8tion.jda.api.entities.TextChannel#deleteMessages(java.util.Collection) TextChannel.deleteMessages(Collection)
     * @see    net.dv8tion.jda.api.entities.MessageChannel#purgeMessages(java.util.List) MessageChannel.purgeMessages(List)
     */
    @Nonnull
    @CheckReturnValue
    AuditableRestAction<Void> delete();

    /**
     * Returns the {@link net.dv8tion.jda.api.JDA JDA} instance related to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return  the corresponding JDA instance
     */
    @Nonnull
    JDA getJDA();

    /**
     * Whether or not this Message has been pinned in its parent channel.
     *
     * @return True - if this message has been pinned.
     */
    boolean isPinned();

    /**
     * Used to add the Message to the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#pinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The pin request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The pin request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The pin request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and:
     *         <br><ul>
     *             <li>Missing {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> pin();

    /**
     * Used to remove the Message from the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#unpinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The unpin request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The unpin request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The unpin request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and:
     *         <br><ul>
     *             <li>Missing {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> unpin();

    /**
     * Adds a reaction to this Message using an {@link net.dv8tion.jda.api.entities.Emote Emote}.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.api.entities.Emote Emote}!</u></b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#REACTION_BLOCKED REACTION_BLOCKED}
     *     <br>The user has blocked the currently logged in account and the reaction failed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#TOO_MANY_REACTIONS TOO_MANY_REACTIONS}
     *     <br>The message already has too many reactions to proceed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}
     *         or {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link net.dv8tion.jda.api.entities.Emote Emote} to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}</li>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is null.</li>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is fake {@link net.dv8tion.jda.api.entities.Emote#isFake() Emote.isFake()}.</li>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} cannot be used in the current channel.
     *                 See {@link Emote#canInteract(User, MessageChannel)} or {@link Emote#canInteract(Member)} for more information.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> addReaction(@Nonnull Emote emote);

    /**
     * Adds a reaction to this Message using a unicode emoji.
     * <br>A reference of unicode emojis can be found here:
     * <a href="http://unicode.org/emoji/charts/full-emoji-list.html" target="_blank">Emoji Table</a>.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <h2>Examples</h2>
     * <code>
     * // custom<br>
     * message.addReaction("minn:245267426227388416").queue();<br>
     * // unicode escape<br>
     * message.addReaction("&#92;uD83D&#92;uDE02").queue();<br>
     * // codepoint notation<br>
     * message.addReaction("U+1F602").queue();
     * </code>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#REACTION_BLOCKED REACTION_BLOCKED}
     *     <br>The user has blocked the currently logged in account and the reaction failed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#TOO_MANY_REACTIONS TOO_MANY_REACTIONS}
     *     <br>The message already has too many reactions to proceed</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode emoji to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}</li>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode emoji is null or empty.
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> addReaction(@Nonnull String unicode);

    /**
     * Removes all reactions from this Message.
     * <br>This is useful for moderator commands that wish to remove all reactions at once from a specific message.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The clear-reactions request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The clear-reactions request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The clear-reactions request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the currently logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the channel.
     * @throws java.lang.IllegalStateException
     *         If this message was <b>not</b> sent in a
     *         {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}.
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> clearReactions();

    /**
     * Removes all reactions for the specified emoji.
     *
     * <h2>Example</h2>
     * <pre><code>
     * // custom
     * message.clearReactions("minn:245267426227388416").queue();
     * // unicode escape
     * message.clearReactions("&#92;uD83D&#92;uDE02").queue();
     * // codepoint notation
     * message.clearReactions("U+1F602").queue();
     * </code></pre>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The currently logged in account lost access to the channel by either being removed from the guild
     *         or losing the {@link net.dv8tion.jda.api.Permission#VIEW_CHANNEL VIEW_CHANNEL} permission</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided unicode emoji doesn't exist. Try using one of the example formats.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The message was deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode emoji to remove reactions for
     *
     * @throws UnsupportedOperationException
     *         If this reaction happened in a private channel
     * @throws InsufficientPermissionException
     *         If the currently logged in account does not have {@link Permission#MESSAGE_MANAGE} in the channel
     * @throws IllegalArgumentException
     *         If provided with null
     *
     * @return {@link RestAction}
     *
     * @since  4.2.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> clearReactions(@Nonnull String unicode);

    /**
     * Removes all reactions for the specified emote.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The currently logged in account lost access to the channel by either being removed from the guild
     *         or losing the {@link net.dv8tion.jda.api.Permission#VIEW_CHANNEL VIEW_CHANNEL} permission</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_EMOJI UNKNOWN_EMOJI}
     *     <br>The provided emote was deleted or doesn't exist.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The message was deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link Emote} to remove reactions for
     *
     * @throws UnsupportedOperationException
     *         If this reaction happened in a private channel
     * @throws InsufficientPermissionException
     *         If the currently logged in account does not have {@link Permission#MESSAGE_MANAGE} in the channel
     * @throws IllegalArgumentException
     *         If provided with null
     *
     * @return {@link RestAction}
     *
     * @since  4.2.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> clearReactions(@Nonnull Emote emote);

    /**
     * Removes a reaction from this Message using an {@link net.dv8tion.jda.api.entities.Emote Emote}.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.api.entities.Emote Emote}!</u></b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link net.dv8tion.jda.api.entities.Emote Emote} to remove as a reaction from this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is null.</li>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} cannot be used in the current channel.
     *                 See {@link Emote#canInteract(User, MessageChannel)} or {@link Emote#canInteract(Member)} for more information.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull Emote emote);

    /**
     * Removes a {@link net.dv8tion.jda.api.entities.User User's} reaction from this Message using an {@link net.dv8tion.jda.api.entities.Emote Emote}.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.api.entities.Emote Emote}!</u></b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when removing the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link net.dv8tion.jda.api.entities.Emote Emote} to remove as a reaction from this Message.
     * @param  user
     *         The {@link net.dv8tion.jda.api.entities.User User} to remove the reaction for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}.
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is null.</li>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is fake {@link net.dv8tion.jda.api.entities.Emote#isFake() Emote.isFake()}.</li>
     *             <li>If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} cannot be used in the current channel.
     *                 See {@link Emote#canInteract(User, MessageChannel)} or {@link Emote#canInteract(Member)} for more information.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull Emote emote, @Nonnull User user);

    /**
     * Removes a reaction from this Message using a unicode emoji.
     * <br>A reference of unicode emojis can be found here:
     * <a href="http://unicode.org/emoji/charts/full-emoji-list.html" target="_blank">Emoji Table</a>.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/unicode.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <h2>Examples</h2>
     * <code>
     * // custom<br>
     * message.removeReaction("minn:245267426227388416").queue();<br>
     * // unicode escape<br>
     * message.removeReaction("&#92;uD83D&#92;uDE02").queue();<br>
     * // codepoint notation<br>
     * message.removeReaction("U+1F602").queue();
     * </code>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode emoji to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode emoji is null or empty.
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull String unicode);

    /**
     * Removes a reaction from this Message using a unicode emoji.
     * <br>A reference of unicode emojis can be found here:
     * <a href="http://unicode.org/emoji/charts/full-emoji-list.html" target="_blank">Emoji Table</a>.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with the same emoji/unicode.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when removing the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode emoji to add as a reaction to this Message.
     * @param  user
     *         The {@link net.dv8tion.jda.api.entities.User User} to remove the reaction for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *             <li>{@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode emoji is null or empty.
     *
     * @return {@link net.dv8tion.jda.api.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    RestAction<Void> removeReaction(@Nonnull String unicode, @Nonnull User user);

    /**
     * This obtains the {@link net.dv8tion.jda.api.entities.User users} who reacted using the given {@link net.dv8tion.jda.api.entities.Emote emote}.
     *
     * <p>Messages maintain a list of reactions, alongside a list of users who added them.
     *
     * <p>Using this data, we can obtain a {@link net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction ReactionPaginationAction}
     * of the users who've reacted to this message.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The retrieve request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link net.dv8tion.jda.api.entities.Emote emote} to retrieve users for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the
     *         logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided {@link net.dv8tion.jda.api.entities.Emote Emote} is null.
     *
     * @return The {@link net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction ReactionPaginationAction} of the emote's users.
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    ReactionPaginationAction retrieveReactionUsers(@Nonnull Emote emote);

    /**
     * This obtains the {@link net.dv8tion.jda.api.entities.User users} who reacted using the given unicode emoji.
     *
     * <p>Messages maintain a list of reactions, alongside a list of users who added them.
     *
     * <p>Using this data, we can obtain a {@link net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction ReactionPaginationAction}
     * of the users who've reacted to this message.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The retrieve request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode emote to retrieve users for.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the
     *         logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode emoji is null or empty.
     *
     * @return The {@link net.dv8tion.jda.api.requests.restaction.pagination.ReactionPaginationAction ReactionPaginationAction} of the emoji's users.
     *
     * @since  4.1.0
     */
    @Nonnull
    @CheckReturnValue
    ReactionPaginationAction retrieveReactionUsers(@Nonnull String unicode);

    /**
     * This obtains the {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} for the given unicode reaction on this message.
     *
     * <p>Messages also store reactions using unicode values.
     *
     * <p>An instance of the related {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} can be
     * obtained through this method by using the emoji's unicode value.
     *
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The unicode value of the reaction emoji.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the
     *         logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode value is null or empty.
     *
     * @return The {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} of this message or null if not present.
     *
     * @since  4.1.0
     */
    @Nullable
    @CheckReturnValue
    MessageReaction.ReactionEmote getReactionByUnicode(@Nonnull String unicode);

    /**
     * This obtains the {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} for the given reaction id on this message.
     *
     * <p>Messages store reactions by keeping a list of reaction names.
     *
     * <p>An instance of the related {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} can be
     * obtained through this method by using the emoji's id.
     *
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  id
     *         The string id of the reaction emoji.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the
     *         logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided id is not a valid snowflake.
     *
     * @return The {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} of this message or null if not present.
     *
     * @since  4.1.0
     */
    @Nullable
    @CheckReturnValue
    MessageReaction.ReactionEmote getReactionById(@Nonnull String id);

    /**
     * This obtains the {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} for the given reaction id on this message.
     *
     * <p>Messages store reactions by keeping a list of reaction names.
     *
     * <p>An instance of the related {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} can be
     * obtained through this method by using the emoji's id.
     *
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  id
     *         The long id of the reaction emoji.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} and the
     *         logged in account does not have {@link net.dv8tion.jda.api.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY} in the channel.
     * @throws java.lang.IllegalArgumentException
     *         If the provided id is not a valid snowflake.
     *
     * @return The {@link net.dv8tion.jda.api.entities.MessageReaction.ReactionEmote ReactionEmote} of this message or null if not present.
     *
     * @since  4.1.0
     */
    @Nullable
    @CheckReturnValue
    MessageReaction.ReactionEmote getReactionById(long id);

    /**
     * Enables/Disables suppression of Embeds on this Message.
     * <br>Suppressing Embeds is equivalent to pressing the {@code X} in the top-right corner of an Embed inside the Discord client.
     *
     * <p>The following {@link net.dv8tion.jda.api.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The clear-reactions request was attempted after the account lost access to the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.api.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.api.entities.Guild Guild}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The suppress-embeds request was attempted after the account lost {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link net.dv8tion.jda.api.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.api.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The suppress-embeds request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  suppressed
     *         Whether or not the embed should be suppressed
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.api.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
     *         and the currently logged in account does not have
     *         {@link net.dv8tion.jda.api.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     * @throws net.dv8tion.jda.api.exceptions.PermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.api.entities.PrivateChannel PrivateChannel}
     *         and the message was not sent by the currently logged in account.
     * @return {@link net.dv8tion.jda.api.requests.restaction.AuditableRestAction AuditableRestAction} - Type: {@link java.lang.Void}
     * @see    #isSuppressedEmbeds()
     */
    @Nonnull
    @CheckReturnValue
    AuditableRestAction<Void> suppressEmbeds(boolean suppressed);

    /**
     * Whether embeds are suppressed for this message.
     * When Embeds are suppressed, they are not displayed on clients nor provided via API until un-suppressed.
     * <br>This is a shortcut method for checking if {@link #getFlags() getFlags()} contains
     * {@link net.dv8tion.jda.api.entities.Message.MessageFlag#EMBEDS_SUPPRESSED MessageFlag#EMBEDS_SUPPRESSED}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @return Whether or not Embeds are suppressed for this Message.
     * @see    #suppressEmbeds(boolean)
     */
    boolean isSuppressedEmbeds();

    /**
     * Returns an EnumSet of all {@link Message.MessageFlag MessageFlags} present for this Message.
     * @return Never-Null EnumSet of present {@link Message.MessageFlag MessageFlags}
     * @see    Message.MessageFlag
     */
    @Nonnull
    EnumSet<MessageFlag> getFlags();

    /**
     * This specifies the {@link net.dv8tion.jda.api.entities.MessageType MessageType} of this Message.
     *
     * <p>Messages can represent more than just simple text sent by Users, they can also be special messages that
     * inform about events that occur. Messages can either be {@link net.dv8tion.jda.api.entities.MessageType#DEFAULT default messages}
     * or special messages like {@link net.dv8tion.jda.api.entities.MessageType#GUILD_MEMBER_JOIN welcome messages}.
     *
     * @return The {@link net.dv8tion.jda.api.entities.MessageType MessageType} of this message.
     */
    @Nonnull
    MessageType getType();

    /**
     * Mention formatting constants, useful for use with {@link java.util.regex.Pattern Patterns}
     */
    enum MentionType
    {
        /**
         * Represents a mention for a {@link net.dv8tion.jda.api.entities.User User}/{@link net.dv8tion.jda.api.entities.Member Member}
         * <br>The first and only group matches the id of the mention.
         */
        USER("<@!?(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.api.entities.Role Role}
         * <br>The first and only group matches the id of the mention.
         */
        ROLE("<@&(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.api.entities.TextChannel TextChannel}
         * <br>The first and only group matches the id of the mention.
         */
        CHANNEL("<#(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.api.entities.Emote Emote}
         * <br>The first group matches the name of the emote and the second the id of the mention.
         */
        EMOTE("<a?:([a-zA-Z0-9_]+):([0-9]+)>"),
        /**
         * Represents a mention for all active users, literal {@code @here}
         */
        HERE("@here"),
        /**
         * Represents a mention for all users in a server, literal {@code @everyone}.
         */
        EVERYONE("@everyone");

        private final Pattern pattern;

        MentionType(String regex)
        {
            this.pattern = Pattern.compile(regex);
        }

        @Nonnull
        public Pattern getPattern()
        {
            return pattern;
        }
    }

    /**
     * Enum representing the flags on a Message.
     * <p>
     * Note: The Values defined in this Enum are not considered final and only represent the current State of <i>known</i> Flags.
     */
    enum MessageFlag
    {
        /**
         * The Message has been published to subscribed Channels (via Channel Following)
         */
        CROSSPOSTED(0),
        /**
         * The Message originated from a Message in another Channel (via Channel Following)
         */
        IS_CROSSPOST(1),
        /**
         * Embeds are suppressed on the Message.
         * @see net.dv8tion.jda.api.entities.Message#isSuppressedEmbeds() Message#isSuppressedEmbeds()
         */
        EMBEDS_SUPPRESSED(2),
        /**
         * Indicates, that the source message of this crosspost was deleted.
         * This should only be possible in combination with {@link #IS_CROSSPOST}
         */
        SOURCE_MESSAGE_DELETED(3),
        /**
         * Indicates, that this Message came from the urgent message system
         */
        URGENT(4);

        private final int value;

        MessageFlag(int offset)
        {
            this.value = 1 << offset;
        }

        /**
         * Returns the value of the MessageFlag as represented in the bitfield. It is always a power of 2 (single bit)
         * @return Non-Zero bit value of the field
         */
        public int getValue()
        {
            return value;
        }

        /**
         * Given a bitfield, this function extracts all Enum values according to their bit values and returns
         * an EnumSet containing all matching MessageFlags
         * @param  bitfield
         *         Non-Negative integer representing a bitfield of MessageFlags
         * @return Never-Null EnumSet of MessageFlags being found in the bitfield
         */
        @Nonnull
        public static EnumSet<MessageFlag> fromBitField(int bitfield)
        {
            Set<MessageFlag> set = Arrays.stream(MessageFlag.values())
                .filter(e -> (e.value & bitfield) > 0)
                .collect(Collectors.toSet());
            return set.isEmpty() ? EnumSet.noneOf(MessageFlag.class) : EnumSet.copyOf(set);
        }

        /**
         * Converts a Collection of MessageFlags back to the integer representing the bitfield.
         * This is the reverse operation of {@link #fromBitField(int)}.
         * @param  coll
         *         A Non-Null Collection of MessageFlags
         * @throws IllegalArgumentException
         *         If the provided Collection is {@code null}
         * @return Integer value of the bitfield representing the given MessageFlags
         */
        public static int toBitField(@Nonnull Collection<MessageFlag> coll)
        {
            Checks.notNull(coll, "Collection");
            int flags = 0;
            for (MessageFlag messageFlag : coll)
            {
                flags |= messageFlag.value;
            }
            return flags;
        }
    }

    /**
     * Represents a {@link net.dv8tion.jda.api.entities.Message Message} file attachment.
     */
    class Attachment implements ISnowflake
    {
        private static final Set<String> IMAGE_EXTENSIONS = new HashSet<>(Arrays.asList("jpg",
                "jpeg", "png", "gif", "webp", "tiff", "svg", "apng"));
        private static final Set<String> VIDEO_EXTENSIONS = new HashSet<>(Arrays.asList("webm",
                "flv", "vob", "avi", "mov", "wmv", "amv", "mp4", "mpg", "mpeg", "gifv"));
        private final long id;
        private final String url;
        private final String proxyUrl;
        private final String fileName;
        private final int size;
        private final int height;
        private final int width;

        private final JDAImpl jda;

        public Attachment(long id, String url, String proxyUrl, String fileName, int size, int height, int width, JDAImpl jda)
        {
            this.id = id;
            this.url = url;
            this.proxyUrl = proxyUrl;
            this.fileName = fileName;
            this.size = size;
            this.height = height;
            this.width = width;
            this.jda = jda;
        }

        /**
         * The corresponding JDA instance for this Attachment
         *
         * @return The corresponding JDA instance for this Attachment
         */
        @Nonnull
        public JDA getJDA()
        {
            return jda;
        }

        @Override
        public long getIdLong()
        {
            return id;
        }

        /**
         * The url of the Attachment, most likely on the Discord servers.
         *
         * @return Non-null String containing the Attachment URL.
         */
        @Nonnull
        public String getUrl()
        {
            return url;
        }

        /**
         * Url to the resource proxied by the Discord CDN.
         *
         * @return Non-null String containing the proxied Attachment url.
         */
        @Nonnull
        public String getProxyUrl()
        {
            return proxyUrl;
        }

        /**
         * The file name of the Attachment when it was first uploaded.
         *
         * @return Non-null String containing the Attachment file name.
         */
        @Nonnull
        public String getFileName()
        {
            return fileName;
        }

        /**
         * The file extension of the Attachment when it was first uploaded.
         * <br>Null is returned if no characters follow the last occurrence of the '{@code .}' character
         * (or if the character is not present in {@link #getFileName()}).
         *
         * @return Non-null String containing the Attachment file extension, or null if it can't be determined.
         */
        @Nullable
        public String getFileExtension()
        {
            int index = fileName.lastIndexOf('.') + 1;
            return index == 0 || index == fileName.length() ? null : fileName.substring(index);
        }

        /**
         * Enqueues a request to retrieve the contents of this Attachment.
         * <br><b>The receiver is expected to close the retrieved {@link java.io.InputStream}.</b>
         *
         * <h2>Example</h2>
         * <pre>{@code
         * public void printContents(Message.Attachment attachment)
         * {
         *     attachment.retrieveInputStream().thenAccept(in -> {
         *         StringBuilder builder = new StringBuilder();
         *         byte[] buf = byte[1024];
         *         int count = 0;
         *         while ((count = in.read(buf)) > 0)
         *         {
         *             builder.append(new String(buf, 0, count));
         *         }
         *         in.close();
         *         System.out.println(builder);
         *     }).exceptionally(t -> { // handle failure
         *         t.printStackTrace();
         *         return null;
         *     });
         * }
         * }</pre>
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.InputStream}
         */
        @Nonnull
        public CompletableFuture<InputStream> retrieveInputStream() // it is expected that the response is closed by the callback!
        {
            CompletableFuture<InputStream> future = new CompletableFuture<>();
            Request req = getRequest();
            OkHttpClient httpClient = getJDA().getHttpClient();
            httpClient.newCall(req).enqueue(FunctionalCallback
                .onFailure((call, e) -> future.completeExceptionally(new UncheckedIOException(e)))
                .onSuccess((call, response) -> {
                    if (response.isSuccessful())
                    {
                        InputStream body = IOUtil.getBody(response);
                        if (!future.complete(body))
                            IOUtil.silentClose(response);
                    }
                    else
                    {
                        future.completeExceptionally(new HttpException(response.code() + ": " + response.message()));
                        IOUtil.silentClose(response);
                    }
                }).build());
            return future;
        }

        /**
         * Downloads the attachment into the current working directory using the file name provided by {@link #getFileName()}.
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h2>Example</h2>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile()
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         */
        @Nonnull
        public CompletableFuture<File> downloadToFile() // using relative path
        {
            return downloadToFile(getFileName());
        }

        /**
         * Downloads the attachment to a file at the specified path (relative or absolute).
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h2>Example</h2>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile("/tmp/" + attachment.getFileName())
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @param  path
         *         The path to save the file to
         *
         * @throws java.lang.IllegalArgumentException
         *         If the provided path is null
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         */
        @Nonnull
        public CompletableFuture<File> downloadToFile(String path)
        {
            Checks.notNull(path, "Path");
            return downloadToFile(new File(path));
        }

        /**
         * Downloads the attachment to a file at the specified path (relative or absolute).
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h2>Example</h2>
         * <pre>{@code
         * public void saveLocally(Message.Attachment attachment)
         * {
         *     attachment.downloadToFile(new File("/tmp/" + attachment.getFileName()))
         *         .thenAccept(file -> System.out.println("Saved attachment to " + file.getName()))
         *         .exceptionally(t ->
         *         { // handle failure
         *             t.printStackTrace();
         *             return null;
         *         });
         * }
         * }</pre>
         *
         * @param  file
         *         The file to write to
         *
         * @throws java.lang.IllegalArgumentException
         *         If the provided file is null or cannot be written to
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link java.io.File}
         */
        @Nonnull
        public CompletableFuture<File> downloadToFile(File file)
        {
            Checks.notNull(file, "File");
            Checks.check(!file.exists() || file.canWrite(), "Cannot write to file %s", file.getName());
            return retrieveInputStream().thenApplyAsync((stream) -> {
                try (FileOutputStream out = new FileOutputStream(file))
                {
                    byte[] buf = new byte[1024];
                    int count;
                    while ((count = stream.read(buf)) > 0)
                    {
                        out.write(buf, 0, count);
                    }
                    return file;
                }
                catch (IOException e)
                {
                    throw new UncheckedIOException(e);
                }
                finally
                {
                    IOUtil.silentClose(stream);
                }
            }, getJDA().getCallbackPool());
        }

        /**
         * Retrieves the image of this attachment and provides an {@link net.dv8tion.jda.api.entities.Icon} equivalent.
         * <br>Useful with {@link net.dv8tion.jda.api.managers.AccountManager#setAvatar(Icon)}.
         * <br>This will download the file using the {@link net.dv8tion.jda.api.JDA#getCallbackPool() callback pool}.
         * Alternatively you can use {@link #retrieveInputStream()} and use a continuation with a different executor.
         *
         * <h2>Example</h2>
         * <pre>{@code
         * public void changeAvatar(Message.Attachment attachment)
         * {
         *     attachment.retrieveAsIcon().thenCompose(icon -> {
         *         SelfUser self = attachment.getJDA().getSelfUser();
         *         AccountManager manager = self.getManager();
         *         return manager.setAvatar(icon).submit();
         *     }).exceptionally(t -> {
         *         t.printStackTrace();
         *         return null;
         *     });
         * }
         * }</pre>
         *
         * @throws java.lang.IllegalStateException
         *         If this is not an image ({@link #isImage()})
         *
         * @return {@link java.util.concurrent.CompletableFuture} - Type: {@link net.dv8tion.jda.api.entities.Icon}
         */
        @Nonnull
        public CompletableFuture<Icon> retrieveAsIcon()
        {
            if (!isImage())
                throw new IllegalStateException("Cannot create an Icon out of this attachment. This is not an image.");
            return retrieveInputStream().thenApplyAsync((stream) ->
            {
                try
                {
                    return Icon.from(stream);
                }
                catch (IOException e)
                {
                    throw new UncheckedIOException(e);
                }
                finally
                {
                    IOUtil.silentClose(stream);
                }
            }, getJDA().getCallbackPool());
        }

        protected Request getRequest()
        {
            return new Request.Builder()
                .url(getUrl())
                .addHeader("user-agent", Requester.USER_AGENT)
                .addHeader("accept-encoding", "gzip, deflate")
                .build();
        }

        /**
         * The size of the attachment in bytes.
         * <br>Example: if {@code getSize()} returns 1024, then the attachment is 1024 bytes, or 1KiB, in size.
         *
         * @return Positive int containing the size of the Attachment.
         */
        public int getSize()
        {
            return size;
        }

        /**
         * The height of the Attachment if this Attachment is an image/video.
         * <br>If this Attachment is neither an image, nor a video, this returns -1.
         *
         * @return int containing image/video Attachment height, or -1 if attachment is neither image nor video.
         */
        public int getHeight()
        {
            return height;
        }

        /**
         * The width of the Attachment if this Attachment is an image/video.
         * <br>If this Attachment is neither an image, nor a video, this returns -1.
         *
         * @return int containing image/video Attachment width, or -1 if attachment is neither image nor video.
         */
        public int getWidth()
        {
            return width;
        }

        /**
         * Whether or not this attachment is an Image,
         * based on {@link #getWidth()}, {@link #getHeight()}, and {@link #getFileExtension()}.
         *
         * @return True if this attachment is an image
         */
        public boolean isImage()
        {
            if (width < 0) return false; //if width is -1, so is height
            String extension = getFileExtension();
            return extension != null && IMAGE_EXTENSIONS.contains(extension.toLowerCase());
        }

        /**
         * Whether or not this attachment is a video,
         * based on {@link #getWidth()}, {@link #getHeight()}, and {@link #getFileExtension()}.
         *
         * @return True if this attachment is a video
         */
        public boolean isVideo()
        {
            if (width < 0) return false; //if width is -1, so is height
            String extension = getFileExtension();
            return extension != null && VIDEO_EXTENSIONS.contains(extension.toLowerCase());
        }
    }
}
