/*
 *     Copyright 2015-2018 Austin Keener & Michael Ritter & Florian Spieß
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.dv8tion.jda.core.entities;

import net.dv8tion.jda.client.entities.Group;
import net.dv8tion.jda.core.JDA;
import net.dv8tion.jda.core.entities.impl.JDAImpl;
import net.dv8tion.jda.core.requests.Requester;
import net.dv8tion.jda.core.requests.RestAction;
import net.dv8tion.jda.core.requests.restaction.AuditableRestAction;
import net.dv8tion.jda.core.requests.restaction.MessageAction;
import net.dv8tion.jda.core.utils.Checks;
import net.dv8tion.jda.core.utils.IOConsumer;
import net.dv8tion.jda.core.utils.IOUtil;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;

import javax.annotation.CheckReturnValue;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.OffsetDateTime;
import java.util.Formattable;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;

/**
 * Represents a Text message received from Discord.
 * <br>This represents messages received from {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannels}.
 *
 * <p><b>This type is not updated. JDA does not keep track of changes to messages, it is advised to do this via events such
 * as {@link net.dv8tion.jda.core.events.message.MessageUpdateEvent MessageUpdateEvent} and similar.</b>
 *
 * <h2>Message Differences</h2>
 * There are 3 implementations of this interface in JDA.
 * <ol>
 *     <li><b>Received Message</b>
 *     <br>Messages received through events or history query.
 *         These messages hold information of <b>existing</b> messages and
 *         can be modified or deleted.</li>
 *     <li><b>System Message</b>
 *     <br>Specification of <b>Received Messages</b> that are generated by Discord
 *         on certain events. Commonly this is used in groups or to indicate a pin within a MessageChannel.
 *         The different types can be found in the {@link net.dv8tion.jda.core.entities.MessageType MessageType} enum.</li>
 *     <li><b>Data Message</b>
 *     <br>This type is produced by {@link net.dv8tion.jda.core.MessageBuilder MessageBuilder}
 *         and only holds sendable information such as content or nonce. These messages do not allow
 *         any modifications via RestActions or information that is generated when sent such as the id to be used.</li>
 * </ol>
 *
 * <p>When a feature is not available it will throw an {@link java.lang.UnsupportedOperationException UnsupportedOperationException}
 * as per interface specifications.
 * <br>Specific operations may have specified information available in the {@code throws} javadoc.
 *
 * <h1>Formattable</h1>
 * This interface extends {@link java.util.Formattable Formattable} and can be used with a {@link java.util.Formatter Formatter}
 * such as used by {@link String#format(String, Object...) String.format(String, Object...)}
 * or {@link java.io.PrintStream#printf(String, Object...) PrintStream.printf(String, Object...)}.
 *
 * <p>This will use {@link #getContentDisplay()} rather than {@link Object#toString()}!
 * <br>Supported Features:
 * <ul>
 *     <li><b>Alternative</b>
 *     <br>   - Using {@link #getContentRaw()}
 *              (Example: {@code %#s} - uses {@link #getContentDisplay()})</li>
 *
 *     <li><b>Width/Left-Justification</b>
 *     <br>   - Ensures the size of a format
 *              (Example: {@code %20s} - uses at minimum 20 chars;
 *              {@code %-10s} - uses left-justified padding)</li>
 *
 *     <li><b>Precision</b>
 *     <br>   - Cuts the content to the specified size
 *              (replacing last 3 chars with {@code ...}; Example: {@code %.20s})</li>
 * </ul>
 *
 * <p>More information on formatting syntax can be found in the {@link java.util.Formatter format syntax documentation}!
 */
public interface Message extends ISnowflake, Formattable
{
    /**
     * The maximum sendable file size (8 MiB)
     *
     * @see net.dv8tion.jda.core.requests.restaction.MessageAction#addFile(java.io.File) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE = 8 << 20;

    /**
     * The maximum sendable file size for nitro (50 MiB)
     *
     * @see net.dv8tion.jda.core.requests.restaction.MessageAction#addFile(java.io.File) MessageAction.addFile(...)
     */
    int MAX_FILE_SIZE_NITRO = 50 << 20;

    /**
     * The maximum amount of files sendable within a single message ({@value})
     *
     * @see net.dv8tion.jda.core.requests.restaction.MessageAction#addFile(java.io.File) MessageAction.addFile(...)
     */
    int MAX_FILE_AMOUNT = 10;

    /**
     * The maximum amount of characters sendable in one message. ({@value})
     * <br>This only applies to the raw content and not embeds!
     *
     * @see net.dv8tion.jda.core.requests.restaction.MessageAction#append(CharSequence) MessageAction.append(...)
     */
    int MAX_CONTENT_LENGTH = 2000;

    /**
     * Pattern used to find instant invites in messages.
     *
     * @see #getInvites() getInvites()
     */
    Pattern INVITE_PATTERN = Pattern.compile("(?:https?://)?discord(?:app\\.com/invite|\\.gg)/([a-z0-9-]+)", Pattern.CASE_INSENSITIVE);

    /**
     * An immutable list of all mentioned {@link net.dv8tion.jda.core.entities.User Users}.
     * <br>If no user was mentioned, this list is empty.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned users
     */
    List<User> getMentionedUsers();

    /**
     * A immutable list of all mentioned {@link net.dv8tion.jda.core.entities.TextChannel TextChannels}.
     * <br>If none were mentioned, this list is empty.
     *
     * <p><b>This may include TextChannels from other {@link net.dv8tion.jda.core.entities.Guild Guilds}</b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned TextChannels
     */
    List<TextChannel> getMentionedChannels();

    /**
     * A immutable list of all mentioned {@link net.dv8tion.jda.core.entities.Role Roles}.
     * <br>If none were mentioned, this list is empty.
     *
     * <p><b>This may include Roles from other {@link net.dv8tion.jda.core.entities.Guild Guilds}</b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return immutable list of mentioned Roles
     */
    List<Role> getMentionedRoles();

    /**
     * Creates an immutable list of {@link net.dv8tion.jda.core.entities.Member Members}
     * representing the users of {@link #getMentionedUsers()} in the specified
     * {@link net.dv8tion.jda.core.entities.Guild Guild}.
     * <br>This is only a convenience method and will skip all users that are not in the specified
     * Guild.
     *
     * @param  guild
     *         Non-null {@link net.dv8tion.jda.core.entities.Guild Guild}
     *         that will be used to retrieve Members.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalArgumentException
     *         If the specified Guild is {@code null}
     *
     * @return Immutable list of mentioned Members
     *
     * @since  3.4.0
     */
    List<Member> getMentionedMembers(Guild guild);

    /**
     * Creates an immutable list of {@link net.dv8tion.jda.core.entities.Member Members}
     * representing the users of {@link #getMentionedUsers()} in the
     * {@link net.dv8tion.jda.core.entities.Guild Guild} this Message was sent in.
     * <br>This is only a convenience method and will skip all users that are not in the specified Guild.
     * <br>It will provide the {@link #getGuild()} output Guild to {@link #getMentionedMembers(Guild)}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If this message was not sent in a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *
     * @return Immutable list of mentioned Members
     *
     * @since  3.4.0
     */
    List<Member> getMentionedMembers();

    /**
     * Combines all instances of {@link net.dv8tion.jda.core.entities.IMentionable IMentionable}
     * filtered by the specified {@link net.dv8tion.jda.core.entities.Message.MentionType MentionType} values.
     * <br>This does not include {@link #getMentionedMembers()} to avoid duplicates.
     *
     * <p>If no MentionType values are given this will fallback to all types.
     *
     * @param  types
     *         Amount of {@link net.dv8tion.jda.core.entities.Message.MentionType MentionTypes}
     *         to include in the list of mentions
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalArgumentException
     *         If provided with {@code null}
     *
     * @return Immutable list of filtered {@link net.dv8tion.jda.core.entities.IMentionable IMentionable} instances
     *
     * @since  3.4.0
     */
    List<IMentionable> getMentions(MentionType... types);

    /**
     * Checks if given {@link net.dv8tion.jda.core.entities.IMentionable IMentionable}
     * was mentioned in this message in any way (@User, @everyone, @here, @Role).
     * <br>If no filtering {@link net.dv8tion.jda.core.entities.Message.MentionType MentionTypes} are
     * specified this will fallback to all mention types.
     *
     * <p>{@link Message.MentionType#HERE MentionType.HERE} and {@link Message.MentionType#EVERYONE MentionType.EVERYONE}
     * will only be checked, if the given {@link net.dv8tion.jda.core.entities.IMentionable IMentionable} is of type
     * {@link net.dv8tion.jda.core.entities.User User} or {@link net.dv8tion.jda.core.entities.Member Member}.
     * <br>Online status of Users/Members is <b>NOT</b> considered when checking {@link Message.MentionType#HERE MentionType.HERE}.
     *
     * @param  mentionable
     *         The mentionable entity to check on.
     * @param  types
     *         The types to include when checking whether this type was mentioned.
     *         This will be used with {@link #getMentions(Message.MentionType...) getMentions(MentionType...)}
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return True, if the given mentionable was mentioned in this message
     */
    boolean isMentioned(IMentionable mentionable, MentionType... types);

    /**
     * Indicates if this Message mentions everyone using @everyone or @here.
     *
     * @return True, if message is mentioning everyone
     */
    boolean mentionsEveryone();

    /**
     * Returns whether or not this Message has been edited before.
     *
     * @return True if this message has been edited.
     */
    boolean isEdited();

    /**
     * Provides the {@link java.time.OffsetDateTime OffsetDateTime} defining when this Message was last
     * edited. If this Message has not been edited ({@link #isEdited()} is {@code false}), then this method
     * will return {@code null}.
     *
     * @return Time of the most recent edit, or {@code null} if the Message has never been edited.
     */
    OffsetDateTime getEditedTime();

    /**
     * The author of this Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return Message author
     */
    User getAuthor();

    /**
     * Returns the author of this Message as a {@link net.dv8tion.jda.core.entities.Member member}.
     * <br>This is just a shortcut to {@link #getGuild()}{@link net.dv8tion.jda.core.entities.Guild#getMember(User) .getMember(getAuthor())}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b> This will return {@code null}
     * if it was not sent from a TextChannel.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return Message author, or {@code null} if the message was not sent from a TextChannel.
     */
    Member getMember();

    /**
     * Returns the jump-to URL for the received message. Clicking this URL in the Discord client will cause the client to
     * jump to the specified message.
     * 
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * 
     * @return A String representing the jump-to URL for the message
     */
    String getJumpUrl();

    /**
     * The textual content of this message in the format that would be shown to the Discord client. All
     * {@link net.dv8tion.jda.core.entities.IMentionable IMentionable} entities will be resolved to the format
     * shown by the Discord client instead of the {@literal <id>} format.
     *
     * <p>This includes resolving:
     * <br>{@link net.dv8tion.jda.core.entities.User Users} / {@link net.dv8tion.jda.core.entities.Member Members}
     * to their @Username/@Nickname format,
     * <br>{@link net.dv8tion.jda.core.entities.TextChannel TextChannels} to their #ChannelName format,
     * <br>{@link net.dv8tion.jda.core.entities.Role Roles} to their @RoleName format
     * <br>{@link net.dv8tion.jda.core.entities.Emote Emotes} (not emojis!) to their {@code :name:} format.
     *
     * <p>If you want the actual Content (mentions as {@literal <@id>}), use {@link #getContentRaw()} instead
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The textual content of the message with mentions resolved to be visually like the Discord client.
     */
    String getContentDisplay();

    /**
     * The raw textual content of this message. Does not resolve {@link net.dv8tion.jda.core.entities.IMentionable IMentionable}
     * entities like {@link #getContentDisplay()} does. This means that this is the completely raw textual content of the message
     * received from Discord and can contain mentions specified by
     * <a href="https://discordapp.com/developers/docs/resources/channel#message-formatting" target="_blank">Discord's Message Formatting</a>.
     *
     * @return The raw textual content of the message, containing unresolved Discord message formatting.
     */
    String getContentRaw();

    /**
     * Gets the textual content of this message using {@link #getContentDisplay()} and then strips it of all markdown characters
     * like {@literal *, **, __, ~~} that provide text formatting. Any characters that match these but are not being used
     * for formatting are escaped to prevent possible formatting.
     *
     * @return The textual content from {@link #getContentDisplay()} with all text formatting characters removed or escaped.
     */
    String getContentStripped();

    /**
     * Creates an immutable List of {@link net.dv8tion.jda.core.entities.Invite Invite} codes
     * that are included in this Message.
     * <br>This will use the {@link java.util.regex.Pattern Pattern} provided
     * under {@link #INVITE_PATTERN} to construct a {@link java.util.regex.Matcher Matcher} that will
     * parse the {@link #getContentRaw()} output and include all codes it finds in a list.
     *
     * <p>You can use the codes to retrieve/validate invites via
     * {@link net.dv8tion.jda.core.entities.Invite#resolve(JDA, String) Invite.resolve(JDA, String)}
     *
     * @return Immutable list of invite codes
     */
    List<String> getInvites();

    /**
     * Validation <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank" >nonce</a> for this Message
     * <br>This can be used to validate that a Message was properly sent to the Discord Service.
     * <br>To set a nonce before sending you may use {@link net.dv8tion.jda.core.MessageBuilder#setNonce(String) MessageBuilder.setNonce(String)}!
     *
     * @return The validation nonce
     *
     * @since  3.4.0
     *
     * @see    net.dv8tion.jda.core.MessageBuilder#setNonce(String)
     * @see    <a href="https://en.wikipedia.org/wiki/Cryptographic_nonce" target="_blank">Cryptographic Nonce - Wikipedia</a>
     */
    String getNonce();

    /**
     * Used to determine if this Message was received from a {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel}
     * of the {@link net.dv8tion.jda.core.entities.ChannelType ChannelType} specified.
     * <br>This will always be false for {@link net.dv8tion.jda.core.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.core.entities.VoiceChannel VoiceChannels}.
     *
     * <p>Useful for restricting functionality to a certain type of channels.
     *
     * @param  type
     *         The {@link net.dv8tion.jda.core.entities.ChannelType ChannelType} to check against.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return True if the {@link net.dv8tion.jda.core.entities.ChannelType ChannelType} which this message was received
     *         from is the same as the one specified by {@code type}.
     */
    boolean isFromType(ChannelType type);

    /**
     * Gets the {@link net.dv8tion.jda.core.entities.ChannelType ChannelType} that this message was received from.
     * <br>This will never be {@link net.dv8tion.jda.core.entities.ChannelType#VOICE} as Messages can't be sent to
     * {@link net.dv8tion.jda.core.entities.VoiceChannel VoiceChannels}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The ChannelType which this message was received from.
     */
    ChannelType getChannelType();

    /**
     * Indicates if this Message was sent by a {@link net.dv8tion.jda.core.entities.Webhook Webhook} instead of a
     * {@link net.dv8tion.jda.core.entities.User User}.
     * <br>Useful if you want to ignore non-users.
     *
     * @return True if this message was sent by a {@link net.dv8tion.jda.core.entities.Webhook Webhook}.
     */
    boolean isWebhookMessage();

    /**
     * Returns the {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel} that this message was sent in.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The MessageChannel of this Message
     */
    MessageChannel getChannel();

    /**
     * Returns the {@link net.dv8tion.jda.core.entities.PrivateChannel PrivateChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a PrivateChannel.</b> This will return {@code null}
     * if it was not sent from a PrivateChannel.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.core.entities.PrivateChannel PrivateChannel}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The PrivateChannel this message was sent in, or {@code null} if it was not sent from a PrivateChannel.
     */
    PrivateChannel getPrivateChannel();

    /**
     * Returns the {@link net.dv8tion.jda.client.entities.Group Group} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a Group.</b> This will return {@code null}
     * if it was not sent from a Group.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.client.entities.Group Group}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The Group this message was sent in, or {@code null} if it was not sent from a Group.
     */
    Group getGroup();

    /**
     * Returns the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} that this message was sent in.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b> This will return {@code null}
     * if it was not sent from a TextChannel.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * <p>Use {@link #getChannel()} for an ambiguous {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel}
     * if you do not need functionality specific to {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The TextChannel this message was sent in, or {@code null} if it was not sent from a TextChannel.
     */
    TextChannel getTextChannel();

    /**
     * The {@link net.dv8tion.jda.core.entities.Category Category} this
     * message was sent in. This will always be {@code null} for DMs and Groups.
     * <br>Equivalent to {@code getTextChannel().getParent()}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return {@link net.dv8tion.jda.core.entities.Category Category} for this message
     */
    Category getCategory();

    /**
     * Returns the {@link net.dv8tion.jda.core.entities.Guild Guild} that this message was sent in.
     * <br>This is just a shortcut to {@link #getTextChannel()}{@link net.dv8tion.jda.core.entities.TextChannel#getGuild() .getGuild()}.
     * <br><b>This is only valid if the Message was actually sent in a TextChannel.</b> This will return {@code null}
     * if it was not sent from a TextChannel.
     * <br>You can check the type of channel this message was sent from using {@link #isFromType(ChannelType)} or {@link #getChannelType()}.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return The Guild this message was sent in, or {@code null} if it was not sent from a TextChannel.
     */
    Guild getGuild();

    /**
     * An unmodifiable list of {@link net.dv8tion.jda.core.entities.Message.Attachment Attachments} that are attached to this message.
     * <br>Most likely this will only ever be 1 {@link net.dv8tion.jda.core.entities.Message.Attachment Attachment} at most.
     *
     * @return Unmodifiable list of {@link net.dv8tion.jda.core.entities.Message.Attachment Attachments}.
     */
    List<Attachment> getAttachments();

    /**
     * An unmodifiable list of {@link net.dv8tion.jda.core.entities.MessageEmbed MessageEmbeds} that are part of this
     * Message.
     *
     * @return Unmodifiable list of all given MessageEmbeds.
     */
    List<MessageEmbed> getEmbeds();

    /**
     * All {@link net.dv8tion.jda.core.entities.Emote Emotes} used in this Message.
     * <br><b>This only includes Custom Emotes, not UTF8 Emojis.</b> JDA classifies Emotes as the Custom Emojis uploaded
     * to a Guild and retrievable with {@link net.dv8tion.jda.core.entities.Guild#getEmotes()}. These are not the same
     * as the UTF8 emojis that Discord also supports.
     * <p>
     * <b>This may or may not contain fake Emotes which means they can be displayed but not used by the logged in account.</b>
     * To check whether an Emote is fake you can test if {@link Emote#isFake()} returns true.
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.core.entities.Emote Emote}!</u></b>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return An immutable list of the Emotes used in this message (example match {@literal <:jda:230988580904763393>})
     */
    List<Emote> getEmotes();

    /**
     * All {@link net.dv8tion.jda.core.entities.MessageReaction MessageReactions} that are on this Message.
     *
     * @return immutable list of all MessageReactions on this message.
     */
    List<MessageReaction> getReactions();

    /**
     * Defines whether or not this Message triggers TTS (Text-To-Speech).
     *
     * @return If this message is TTS.
     */
    boolean isTTS();

    /**
     * Edits this Message's content to the provided String.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the
     *         {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account, or if
     *         {@code newContent}'s length is 0 or greater than 2000.
     *
     * @return {@link net.dv8tion.jda.core.requests.restaction.MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.core.entities.Message Message} with the updated content
     */
    @CheckReturnValue
    MessageAction editMessage(CharSequence newContent);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.core.entities.MessageEmbed MessageEmbed}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the
     *         {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account, or
     *         if the passed-in embed is {@code null}
     *         or not {@link net.dv8tion.jda.core.entities.MessageEmbed#isSendable(net.dv8tion.jda.core.AccountType) sendable}
     *
     * @return {@link net.dv8tion.jda.core.requests.restaction.MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.core.entities.Message Message} with the updated content
     */
    @CheckReturnValue
    MessageAction editMessage(MessageEmbed newContent);

    /**
     * Edits this Message's content to the provided format.
     * <br>Shortcut for {@link net.dv8tion.jda.core.MessageBuilder#appendFormat(String, Object...)}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the
     *         {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  format
     *         Format String used to generate the Message's content via
     *         {@link net.dv8tion.jda.core.MessageBuilder#appendFormat(String, Object...)} specification
     * @param  args
     *         The arguments to use in order to be converted in the format string
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws IllegalArgumentException
     *         If the provided format String is {@code null} or blank, or if
     *         the created message exceeds the 2000 character limit
     * @throws java.util.IllegalFormatException
     *         If a format string contains an illegal syntax,
     *         a format specifier that is incompatible with the given arguments,
     *         insufficient arguments given the format string, or other illegal conditions.
     *         For specification of all possible formatting errors,
     *         see the <a href="../util/Formatter.html#detail">Details</a>
     *         section of the formatter class specification.
     * @throws IllegalStateException
     *         If the message attempting to be edited was not created by the currently logged in account
     *
     * @return {@link net.dv8tion.jda.core.requests.restaction.MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.core.entities.Message Message} with the updated content
     */
    @CheckReturnValue
    MessageAction editMessageFormat(String format, Object... args);

    /**
     * Edits this Message's content to the provided {@link net.dv8tion.jda.core.entities.Message Message}.
     * <br><b>Messages can only be edited by the account that sent them!</b>.
     *
     * <p>This message instance will not be updated by this operation, please use the response message instead.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The edit was attempted after the account lost access to the
     *         {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The edit was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_WRITE Permission.MESSAGE_WRITE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *     <br>The edit was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  newContent
     *         the new content of the Message
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws java.lang.IllegalStateException
     *         <ul>
     *             <li>If the message attempting to be edited was not created by the currently logged in account</li>
     *             <li>If the message contains a MessageEmebd that is not
     *                 {@link net.dv8tion.jda.core.entities.MessageEmbed#isSendable(net.dv8tion.jda.core.AccountType) sendable}</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.core.requests.restaction.MessageAction MessageAction}
     *         <br>The {@link net.dv8tion.jda.core.entities.Message Message} with the updated content
     */
    @CheckReturnValue
    MessageAction editMessage(Message newContent);

    /**
     * Deletes this Message from Discord.
     * <br>If this Message was not sent by the currently logged in account, then this will fail unless the Message is from
     * a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} and the current account has
     * {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     *
     * <p><u>To delete many messages at once in a {@link net.dv8tion.jda.core.entities.MessageChannel MessageChannel}
     * you should use {@link net.dv8tion.jda.core.entities.MessageChannel#purgeMessages(List) MessageChannel.purgeMessages(Collection)} instead.</u>
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The delete was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The delete was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} when deleting another Member's message
     *         or lost {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The pin was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is a Data Message (output of {@link net.dv8tion.jda.core.MessageBuilder MessageBuilder})
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If this Message was not sent by the currently logged in account, the Message was sent in a
     *         {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}, and the currently logged in account
     *         does not have {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the channel.
     * @throws java.lang.IllegalStateException
     *         If this Message was not sent by the currently logged in account and it was <b>not</b> sent in a
     *         {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.
     *
     * @return {@link net.dv8tion.jda.core.requests.restaction.AuditableRestAction AuditableRestAction}
     *
     * @see    net.dv8tion.jda.core.entities.TextChannel#deleteMessages(java.util.Collection) TextChannel.deleteMessages(Collection)
     * @see    net.dv8tion.jda.core.entities.MessageChannel#purgeMessages(java.util.List) MessageChannel.purgeMessages(List)
     */
    @CheckReturnValue
    AuditableRestAction<Void> delete();

    /**
     * Returns the {@link net.dv8tion.jda.core.JDA JDA} instance related to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     *
     * @return  the corresponding JDA instance
     */
    JDA getJDA();

    /**
     * Whether or not this Message has been pinned in its parent channel.
     *
     * @return True - if this message has been pinned.
     */
    boolean isPinned();

    /**
     * Used to add the Message to the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#pinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The pin request was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The pin request was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The pin request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} and:
     *         <br><ul>
     *             <li>Missing {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.core.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @CheckReturnValue
    RestAction<Void> pin();

    /**
     * Used to remove the Message from the {@link #getChannel() MessageChannel's} pinned message list.
     * <br>This is a shortcut method to {@link MessageChannel#unpinMessageById(String)}.
     *
     * <p>The success or failure of this action will not affect the return of {@link #isPinned()}.
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The unpin request was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The unpin request was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in
     *         the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The unpin request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If this Message is from a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} and:
     *         <br><ul>
     *             <li>Missing {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ}.
     *             <br>The account needs access the the channel to pin a message in it.</li>
     *             <li>Missing {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}.
     *             <br>Required to actually pin the Message.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.core.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @CheckReturnValue
    RestAction<Void> unpin();

    /**
     * Adds a reaction to this Message using an {@link net.dv8tion.jda.core.entities.Emote Emote}.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p><b><u>Unicode emojis are not included as {@link net.dv8tion.jda.core.entities.Emote Emote}!</u></b>
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.core.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}
     *         in the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  emote
     *         The {@link net.dv8tion.jda.core.entities.Emote Emote} to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link net.dv8tion.jda.core.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}</li>
     *             <li>{@link net.dv8tion.jda.core.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         <ul>
     *             <li>If the provided {@link net.dv8tion.jda.core.entities.Emote Emote} is null.</li>
     *             <li>If the provided {@link net.dv8tion.jda.core.entities.Emote Emote} is fake {@link net.dv8tion.jda.core.entities.Emote#isFake() Emote.isFake()}.</li>
     *             <li>If the provided {@link net.dv8tion.jda.core.entities.Emote Emote} cannot be used in the current channel.
     *                 See {@link Emote#canInteract(User, MessageChannel)} or {@link Emote#canInteract(Member)} for more information.</li>
     *         </ul>
     *
     * @return {@link net.dv8tion.jda.core.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @CheckReturnValue
    RestAction<Void> addReaction(Emote emote);

    /**
     * Adds a reaction to this Message using a UTF8 emoji.
     * <br>A reference of UTF8 emojis can be found here:
     * <a href="http://unicode.org/emoji/charts/full-emoji-list.html" target="_blank">Emoji Table</a>.
     *
     * <p>This message instance will not be updated by this operation.
     *
     * <p>Reactions are the small emoji/emotes below a message that have a counter beside them
     * showing how many users have reacted with same emoji/emote.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The reaction request was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.
     *     <br>Also can happen if the account lost the {@link net.dv8tion.jda.core.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The reaction request was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}
     *         in the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The reaction request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @param  unicode
     *         The UTF8 emoji to add as a reaction to this Message.
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         and the logged in account does not have
     *         <ul>
     *             <li>{@link net.dv8tion.jda.core.Permission#MESSAGE_ADD_REACTION Permission.MESSAGE_ADD_REACTION}</li>
     *             <li>{@link net.dv8tion.jda.core.Permission#MESSAGE_HISTORY Permission.MESSAGE_HISTORY}</li>
     *         </ul>
     * @throws java.lang.IllegalArgumentException
     *         If the provided unicode emoji is null or empty.
     *
     * @return {@link net.dv8tion.jda.core.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @CheckReturnValue
    RestAction<Void> addReaction(String unicode);

    /**
     * Removes all reactions from this Message.
     * <br>This is useful for moderator commands that wish to remove all reactions at once from a specific message.
     *
     * <p><b>Neither success nor failure of this request will affect this Message's {@link #getReactions()} return as Message is immutable.</b>
     *
     * <p>The following {@link net.dv8tion.jda.core.requests.ErrorResponse ErrorResponses} are possible:
     * <ul>
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_ACCESS MISSING_ACCESS}
     *     <br>The clear-reactions request was attempted after the account lost access to the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         due to {@link net.dv8tion.jda.core.Permission#MESSAGE_READ Permission.MESSAGE_READ} being revoked, or the
     *         account lost access to the {@link net.dv8tion.jda.core.entities.Guild Guild} or {@link net.dv8tion.jda.client.entities.Group Group}
     *         typically due to being kicked or removed.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#MISSING_PERMISSIONS MISSING_PERMISSIONS}
     *     <br>The clear-reactions request was attempted after the account lost {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE}
     *         in the {@link net.dv8tion.jda.core.entities.TextChannel TextChannel} when adding the reaction.</li>
     *
     *     <li>{@link net.dv8tion.jda.core.requests.ErrorResponse#UNKNOWN_MESSAGE UNKNOWN_MESSAGE}
     *         The clear-reactions request was attempted after the Message had been deleted.</li>
     * </ul>
     *
     * @throws java.lang.UnsupportedOperationException
     *         If this is not a Received Message from {@link net.dv8tion.jda.core.entities.MessageType#DEFAULT MessageType.DEFAULT}
     * @throws net.dv8tion.jda.core.exceptions.InsufficientPermissionException
     *         If the MessageChannel this message was sent in was a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
     *         and the currently logged in account does not have
     *         {@link net.dv8tion.jda.core.Permission#MESSAGE_MANAGE Permission.MESSAGE_MANAGE} in the channel.
     * @throws java.lang.IllegalStateException
     *         If this message was <b>not</b> sent in a
     *         {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}.
     * @return {@link net.dv8tion.jda.core.requests.RestAction RestAction} - Type: {@link java.lang.Void}
     */
    @CheckReturnValue
    RestAction<Void> clearReactions();

    /**
     * This specifies the {@link net.dv8tion.jda.core.entities.MessageType MessageType} of this Message.
     *
     * <p>Messages can represent more than just simple text sent by Users, they can also be special messages that
     * inform about events occurs. A few examples are the system message informing that a message has been pinned.
     * Another would be the system message informing that a call has been started or ended in a group.
     *
     * @return The {@link net.dv8tion.jda.core.entities.MessageType MessageType} of this message.
     */
    MessageType getType();

    /**
     * Mention formatting constants, useful for use with {@link java.util.regex.Pattern Patterns}
     */
    enum MentionType
    {
        /**
         * Represents a mention for a {@link net.dv8tion.jda.core.entities.User User}/{@link net.dv8tion.jda.core.entities.Member Member}
         */
        USER("<@!?(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.core.entities.Role Role}
         */
        ROLE("<@&(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.core.entities.TextChannel TextChannel}
         */
        CHANNEL("<#(\\d+)>"),
        /**
         * Represents a mention for a {@link net.dv8tion.jda.core.entities.Emote Emote}
         */
        EMOTE("<a?:([a-zA-Z0-9_]+):([0-9]+)>"),
        /**
         * Represents a mention for all active users, literal {@code @here}
         */
        HERE("@here"),
        /**
         * Represents a mention for all users in a server, literal {@code @everyone}.
         *
         * <p>This is not the same as {@code guild.getPublicRole().getAsMention()}!
         */
        EVERYONE("@everyone");

        private final Pattern pattern;

        MentionType(String regex)
        {
            this.pattern = Pattern.compile(regex);
        }

        public Pattern getPattern()
        {
            return pattern;
        }
    }

    /**
     * Represents a {@link net.dv8tion.jda.core.entities.Message Message} file attachment.
     */
    class Attachment implements ISnowflake
    {
        private final long id;
        private final String url;
        private final String proxyUrl;
        private final String fileName;
        private final int size;
        private final int height;
        private final int width;

        private final JDAImpl jda;

        public Attachment(long id, String url, String proxyUrl, String fileName, int size, int height, int width, JDAImpl jda)
        {
            this.id = id;
            this.url = url;
            this.proxyUrl = proxyUrl;
            this.fileName = fileName;
            this.size = size;
            this.height = height;
            this.width = width;
            this.jda = jda;
        }

        /**
         * The corresponding JDA instance for this Attachment
         *
         * @return The corresponding JDA instance for this Attachment
         */
        public JDA getJDA()
        {
            return jda;
        }

        @Override
        public long getIdLong()
        {
            return id;
        }

        /**
         * The url of the Attachment, most likely on the Discord servers.
         *
         * @return Non-null String containing the Attachment URL.
         */
        public String getUrl()
        {
            return url;
        }

        /**
         * The url of the Attachment, proxied by Discord.
         * <br>Url to the resource proxied by https://images.discordapp.net
         * <br><b>Note: </b> This URL will most likely only work for images. ({@link #isImage()})
         *
         * @return Non-null String containing the proxied Attachment url.
         */
        public String getProxyUrl()
        {
            return proxyUrl;
        }

        /**
         * The file name of the Attachment when it was first uploaded.
         *
         * @return Non-null String containing the Attachment file name.
         */
        public String getFileName()
        {
            return fileName;
        }

        /**
         * Creates an {@link net.dv8tion.jda.core.entities.Icon Icon} instance for
         * this attachment if {@link #isImage() isImage()} is {@code true}!
         * <br>This is a convenience method that can be used to retrieve an Icon from an attachment image link which
         * requires a set user-agent to be loaded.
         *
         * <p>When a global proxy was specified via {@link net.dv8tion.jda.core.JDABuilder JDABuilder} this will use the
         * specified proxy to create an {@link java.io.InputStream InputStream} otherwise it will use a normal {@link java.net.URLConnection URLConnection}
         * with the User-Agent for the currently logged in account.
         *
         * @throws IOException
         *         If an IOError occurs while reading the image
         * @throws java.lang.IllegalStateException
         *         If this is not an image attachment
         *
         * @return {@link net.dv8tion.jda.core.entities.Icon Icon} for this image attachment
         *
         * @since  3.4.0
         */
        public Icon getAsIcon() throws IOException
        {
            if (!isImage())
                throw new IllegalStateException("Cannot create an Icon out of this attachment. This is not an image.");
            AtomicReference<Icon> icon = new AtomicReference<>();
            withInputStream((in) -> icon.set(Icon.from(in)));
            return icon.get();
        }

        /**
         * Downloads this attachment to given File
         *
         * @param  file
         *         The file, where the attachment will get downloaded to
         *
         * @return boolean true, if successful, otherwise false
         */
        public boolean download(File file)
        {
            try
            {
                withInputStream((in) -> Files.copy(in, Paths.get(file.getAbsolutePath())));
                return true;
            }
            catch (Exception e)
            {
                JDAImpl.LOG.error("Error while downloading an attachment", e);
            }
            return false;
        }

        /**
         * Creates a copy of the {@link java.io.InputStream InputStream} that is created using an {@link okhttp3.OkHttpClient OkHttpClient}.
         *
         * <p>You can access the input stream directly using {@link #withInputStream(net.dv8tion.jda.core.utils.IOConsumer) withInputStream(IOConsumer)}
         * which will have an open input stream available within the consumer scope. The stream will be closed once that method returns.
         *
         * @throws java.io.IOException
         *         If an IO error occurs trying to read from the opened HTTP channel
         *
         * @return InputStream copy of the response body for this Attachment
         *
         * @since  3.4.0
         */
        public InputStream getInputStream() throws IOException
        {
            try (Response response = openConnection())
            {
                // creates a copy in order to properly close the response
                InputStream in = Requester.getBody(response);
                return new ByteArrayInputStream(IOUtil.readFully(in));
            }
        }

        /**
         * Allows to access the InputStream that is available from the HTTP {@link okhttp3.Response Response}
         * to be used without having to copy it.
         * <br>Unlike {@link #getInputStream()} this does not return a full copy of the input stream.
         * Instead this method will provide the InputStream data in the specified consumer in which it is still accessible.
         *
         * <p><b>When this method returns the InputStream will be closed accordingly!</b>
         *
         * @param  then
         *         Not-null {@link net.dv8tion.jda.core.utils.IOConsumer IOConsumer} to accept the InputStream
         *
         * @throws java.lang.IllegalArgumentException
         *         If the provided IOConsumer is {@code null}
         * @throws IOException
         *         If an IOException occurs within the IOConsumer or while opening an HTTP channel
         *
         * @since  3.4.0
         */
        public void withInputStream(IOConsumer<InputStream> then) throws IOException
        {
            Checks.notNull(then, "Consumer");
            try (Response response = openConnection())
            {
                then.accept(Requester.getBody(response));
            }
        }

        protected Response openConnection() throws IOException
        {
            final OkHttpClient client = jda.getRequester().getHttpClient();
            final Request request = new Request.Builder().url(getUrl())
                        .addHeader("user-agent", Requester.USER_AGENT)
                        .addHeader("accept-encoding", "gzip")
                        .build();
            return client.newCall(request).execute();
        }

        /**
         * The size of the attachment in bytes.
         * <br>Example: if {@code getSize()} returns 1024, then the attachment is 1024 bytes, or 1KiB, in size.
         *
         * @return Positive int containing the size of the Attachment.
         */
        public int getSize()
        {
            return size;
        }

        /**
         * The height of the Attachment if this Attachment is an image.
         * <br>If this Attachment is not an image, this returns -1.
         *
         * @return int containing image Attachment height.
         */
        public int getHeight()
        {
            return height;
        }

        /**
         * The width of the Attachment if this Attachment is an image.
         * <br>If this Attachment is not an image, this returns -1.
         *
         * @return int containing image Attachment width.
         */
        public int getWidth()
        {
            return width;
        }

        /**
         * Whether or not this attachment is an Image.
         * <br>Based on the values of getHeight and getWidth being larger than zero.
         *
         * @return True if width and height are greater than zero.
         */
        public boolean isImage()
        {
            return height > 0 && width > 0;
        }
    }
}
