package me.saket.telephoto.subsamplingimage.internal

import android.graphics.BitmapFactory
import android.graphics.BitmapRegionDecoder
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.graphics.toAndroidRect
import androidx.compose.ui.unit.IntOffset
import androidx.compose.ui.unit.IntRect
import androidx.compose.ui.unit.IntSize
import androidx.tracing.trace
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExecutorCoroutineDispatcher
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.withContext
import me.saket.telephoto.subsamplingimage.ImageBitmapOptions
import me.saket.telephoto.subsamplingimage.SubSamplingImageSource
import me.saket.telephoto.subsamplingimage.internal.ExifMetadata.ImageOrientation
import me.saket.telephoto.subsamplingimage.toAndroidConfig

/** Bitmap decoder backed by Android's [BitmapRegionDecoder]. */
internal class AndroidImageRegionDecoder private constructor(
  private val imageSource: SubSamplingImageSource,
  private val imageOptions: ImageBitmapOptions,
  private val decoder: BitmapRegionDecoder,
  private val exif: ExifMetadata,
  private val dispatcher: ExecutorCoroutineDispatcher,
) : ImageRegionDecoder {

  override val imageSize: IntSize = decoder.size()
  override val imageOrientation: ImageOrientation get() = exif.orientation

  override suspend fun decodeRegion(region: BitmapRegionTile): ImageBitmap {
    val options = BitmapFactory.Options().apply {
      inSampleSize = region.sampleSize.size
      inPreferredConfig = imageOptions.config.toAndroidConfig()
    }

    val bounds = region.bounds.rotateBy(
      // Rotation of images is transparent to consumers of ImageRegionDecoder.
      // When tiles are generated by SubSamplingImage, it assumes that the image is
      // already in the correct orientation. To counter this, rotate the bounds in
      // anticlockwise direction to get the bounds in the original non-rotated image.
      degrees = -exif.orientation.degrees,
      unRotatedParent = IntRect(offset = IntOffset.Zero, size = decoder.size())
    )

    val bitmap = withContext(dispatcher) {
      trace("decodeRegion") {
        decoder.decodeRegion(bounds.toAndroidRect(), options)?.asImageBitmap()
      }
    }
    return checkNotNull(bitmap) {
      "BitmapRegionDecoder returned a null bitmap. Image format may not be supported: $imageSource."
    }
  }

  override fun recycle() {
    // FYI BitmapRegionDecoder's documentation says explicit recycling is not needed,
    // but that is a lie. Instrumentation tests for SubSamplingImage() on API 31 run into
    // low memory because the native state of decoders aren't cleared after each test,
    // causing Android to panic and kill all processes (including the test).
    decoder.recycle()
    dispatcher.close()
  }

  private fun BitmapRegionDecoder.size(): IntSize {
    val shouldFlip = when (exif.orientation) {
      ImageOrientation.Orientation90,
      ImageOrientation.Orientation270 -> true
      else -> false
    }

    return IntSize(
      width = if (shouldFlip) height else width,
      height = if (shouldFlip) width else height,
    )
  }

  companion object {
    @OptIn(DelicateCoroutinesApi::class)
    val Factory = ImageRegionDecoder.Factory { params ->
      val dispatcher = newSingleThreadContext("AndroidImageRegionDecoder")

      AndroidImageRegionDecoder(
        imageSource = params.imageSource,
        imageOptions = params.imageOptions,
        decoder = withContext(dispatcher) {
          params.imageSource.decoder(params.context)
        },
        exif = params.exif,
        dispatcher = dispatcher,
      )
    }
  }
}
