package ltd.dolink.arch.adapter;

import androidx.annotation.NonNull;
import androidx.lifecycle.Lifecycle;
import androidx.lifecycle.Lifecycle.Event;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.LifecycleRegistry;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;
import androidx.viewbinding.ViewBinding;

import java.lang.reflect.InvocationTargetException;
import java.util.Objects;

import ltd.dolink.arch.State;

public abstract class CellViewHolder<VB extends ViewBinding, STATE extends State> extends ViewHolder implements CellView<STATE> {
    private final VB viewBinding;
    private final LifecycleOwner lifecycleOwner = new LifecycleOwner() {
        private final LifecycleRegistry lifecycleRegistry = new LifecycleRegistry(this);

        @NonNull
        @Override
        public Lifecycle getLifecycle() {
            return lifecycleRegistry;
        }
    };
    private CellAdapter adapter;

    public CellViewHolder(@NonNull VB viewBinding) {
        super(viewBinding.getRoot());
        this.viewBinding = viewBinding;
    }

    protected CellAdapter getAdapter() {
        return adapter;
    }

    CellViewHolder<VB, STATE> setAdapter(@NonNull CellAdapter adapter) {
        this.adapter = adapter;
        return this;
    }

    protected void onViewAttachedToWindow() {
        if (lifecycleOwner.getLifecycle() instanceof LifecycleRegistry) {
            LifecycleRegistry lifecycleRegistry = (LifecycleRegistry) lifecycleOwner.getLifecycle();
            lifecycleRegistry.handleLifecycleEvent(Event.ON_CREATE);
        }
    }

    protected void onViewDetachedFromWindow() {
        if (lifecycleOwner.getLifecycle() instanceof LifecycleRegistry) {
            LifecycleRegistry lifecycleRegistry = (LifecycleRegistry) lifecycleOwner.getLifecycle();
            lifecycleRegistry.handleLifecycleEvent(Event.ON_DESTROY);
        }
    }

    protected void onViewRecycled() {

    }

    protected boolean onFailedToRecycleView() {
        return false;
    }

    public VB getViewBinding() {
        return viewBinding;
    }

    @NonNull
    @Override
    public LifecycleOwner getLifecycleOwner() {
        return lifecycleOwner;
    }

    public interface Factory<VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> {
        VH create(@NonNull VB viewBinding);
    }

    public static class DefaultFactory<VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> implements Factory<VB, STATE, VH> {
        @NonNull
        private final Class<VH> viewHolderClass;

        public DefaultFactory(@NonNull Class<VH> viewHolderClass) {
            Objects.requireNonNull(viewHolderClass);
            this.viewHolderClass = viewHolderClass;
        }


        @Override
        public VH create(@NonNull VB viewBinding) {
            try {
                return viewHolderClass.getConstructor(viewBinding.getClass()).newInstance(viewBinding);
            } catch (IllegalAccessException | InvocationTargetException | InstantiationException | NoSuchMethodException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
