package ltd.dolink.arch.adapter;


import android.view.LayoutInflater;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.viewbinding.ViewBinding;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import ltd.dolink.arch.adapter.CellViewHolder.DefaultFactory;
import ltd.dolink.arch.adapter.CellViewHolder.Factory;
import ltd.dolink.arch.viewbinding.ViewBindingFactory;

public abstract class CellBinder {
    private final Map<Integer, ViewHolderFactoryBoundViewBinding> mapper = new HashMap<>();

    public abstract CellState getItem(int position);

    public abstract int getItemCount();


    public <VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> void link(int cellType, @NonNull Class<STATE> stateClass, @NonNull Class<VB> viewBindingClass, @NonNull Class<VH> viewHolderClass) {
        link(cellType, stateClass, ViewBindingFactory.of(viewBindingClass), new DefaultFactory<>(viewHolderClass));
    }

    public <VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> void link(int cellType, @NonNull Class<STATE> stateClass, @NonNull Class<VB> viewBindingClass, @NonNull Factory<VB, STATE, VH> viewHolderFactory) {
        link(cellType, stateClass, ViewBindingFactory.of(viewBindingClass), viewHolderFactory);
    }

    public <VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> void link(int cellType, @NonNull Class<STATE> stateClass, @NonNull ViewBindingFactory<VB> viewBindingFactory, @NonNull Class<VH> viewHolderClass) {
        link(cellType, stateClass, viewBindingFactory, new DefaultFactory<>(viewHolderClass));
    }

    public <VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> void link(int cellType, @NonNull Class<STATE> stateClass, @NonNull ViewBindingFactory<VB> viewBindingFactory, @NonNull Factory<VB, STATE, VH> viewHolderFactory) {
        Objects.requireNonNull(stateClass);
        mapper.put(cellType, new ViewHolderFactoryBoundViewBinding<>(viewBindingFactory, viewHolderFactory));
    }

    protected <VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> VH createViewHolder(@NonNull ViewGroup parent, int cellType) {
        Objects.requireNonNull(parent);
        ViewHolderFactoryBoundViewBinding<VB, STATE, VH> viewHolderFactory = mapper.get(cellType);

        VB viewBinding = viewHolderFactory.viewBindingFactory.inflate(LayoutInflater.from(parent.getContext()), parent, false);
        return viewHolderFactory.create(viewBinding);
    }

    public static class ListCellBinder<STATE extends CellState> extends CellBinder {
        @NonNull
        private final List<STATE> data;

        public ListCellBinder(@NonNull List<STATE> data) {
            Objects.requireNonNull(data);
            this.data = data;
        }

        @NonNull
        public List<STATE> getData() {
            return data;
        }

        @Override
        public STATE getItem(int position) {
            return data.get(position);
        }

        @Override
        public int getItemCount() {
            return data.size();
        }
    }

    public static class ArrayBinder<STATE extends CellState> extends CellBinder {
        @NonNull
        private final STATE[] data;

        public ArrayBinder(@NonNull STATE[] data) {
            Objects.requireNonNull(data);
            this.data = data;
        }

        @NonNull
        public STATE[] getData() {
            return data;
        }

        @Override
        public STATE getItem(int position) {
            return data[position];
        }

        @Override
        public int getItemCount() {
            return data.length;
        }
    }

    class ViewHolderFactoryBoundViewBinding<VB extends ViewBinding, STATE extends CellState, VH extends CellViewHolder<VB, STATE>> implements Factory<VB, STATE, VH> {
        private final ViewBindingFactory<VB> viewBindingFactory;
        private final Factory<VB, STATE, VH> viewHolderFactory;

        ViewHolderFactoryBoundViewBinding(@NonNull ViewBindingFactory<VB> viewBindingFactory, @NonNull Factory<VB, STATE, VH> viewHolderFactory) {
            Objects.requireNonNull(viewBindingFactory);
            Objects.requireNonNull(viewHolderFactory);
            this.viewBindingFactory = viewBindingFactory;
            this.viewHolderFactory = viewHolderFactory;
        }


        @Override
        public VH create(@NonNull VB viewBinding) {
            return viewHolderFactory.create(viewBinding);
        }
    }
}
