/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-api-contract-js/parameter_validation_rule */
var utils = require('vertx-js/util/utils');
var ParameterTypeValidator = require('vertx-web-api-contract-js/parameter_type_validator');
var RequestParameter = require('vertx-web-api-contract-js/request_parameter');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JParameterValidationRule = Java.type('io.vertx.ext.web.api.validation.ParameterValidationRule');

/**
 This function is an inner wrapper for ParameterTypeValidator inside ValidationHandler parameter maps. <b>Don't
 instantiate this class</b>, if you want to add custom ParameterTypeValidator to a parameter use functions in
 @class
*/
var ParameterValidationRule = function(j_val) {

  var j_parameterValidationRule = j_val;
  var that = this;

  /**
   This function return the name of the parameter expected into parameter lists

   @public

   @return {string} name of the parameter
   */
  this.getName = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_parameterValidationRule["getName()"]();
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   This function will be called when there is only a string as parameter. It will throw a ValidationError in an
   error during validation occurs

   @public
   @param value {string} list of values that will be validated 
   @return {RequestParameter}
   */
  this.validateSingleParam = function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'string') {
      return utils.convReturnVertxGen(RequestParameter, j_parameterValidationRule["validateSingleParam(java.lang.String)"](value));
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   This function will be called when there is a List<String> that need to be validated. It must check if array is
   expected or not. It will throw a ValidationError in an error during validation occurs

   @public
   @param value {Array.<string>} list of values that will be validated 
   @return {RequestParameter}
   */
  this.validateArrayParam = function(value) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'object' && __args[0] instanceof Array) {
      return utils.convReturnVertxGen(RequestParameter, j_parameterValidationRule["validateArrayParam(java.util.List)"](utils.convParamListBasicOther(value)));
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return true if parameter is optional

   @public

   @return {boolean} true if is optional, false otherwise
   */
  this.isOptional = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_parameterValidationRule["isOptional()"]();
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Return ParameterTypeValidator instance used inside this parameter validation rule

   @public

   @return {ParameterTypeValidator} 
   */
  this.parameterTypeValidator = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(ParameterTypeValidator, j_parameterValidationRule["parameterTypeValidator()"]());
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   allowEmptyValue is used in query, header, cookie and form parameters. This is its behaviour:
   <ol>
   <li>During validation, the ValidationHandler check if there's a parameter with combination of location and name
   as defined in this rule </li>
   <li>If it not exists, It will check allowEmptyValue and if there's a default value set inside
   ParameterTypeValidator:</li>
   <ul>
   <li>If this condition it's true, It marks as validated the parameter and returns the default value (inside
   RequestParameter)</li>
   <li>If this condition it's false, It throws ValidationException</li>
   </ul>
   <li>If the parameter exists, It checks if parameter is null or empty string:</li>
   <ul>
   <li>If allowEmptyValue it's true, It marks as validated the parameter and returns the default value if it exists
   (inside RequestParameter)</li>
   <li>If allowEmptyValue it's false, It throws ValidationException</li>
   </ul>
   </ol>

   @public

   @return {boolean} value of allowEmptyValue
   */
  this.allowEmptyValue = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return j_parameterValidationRule["allowEmptyValue()"]();
    } else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_parameterValidationRule;
};

ParameterValidationRule._jclass = utils.getJavaClass("io.vertx.ext.web.api.validation.ParameterValidationRule");
ParameterValidationRule._jtype = {
  accept: function(obj) {
    return ParameterValidationRule._jclass.isInstance(obj._jdel);
  },
  wrap: function(jdel) {
    var obj = Object.create(ParameterValidationRule.prototype, {});
    ParameterValidationRule.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
ParameterValidationRule._create = function(jdel) {
  var obj = Object.create(ParameterValidationRule.prototype, {});
  ParameterValidationRule.apply(obj, arguments);
  return obj;
}
module.exports = ParameterValidationRule;