/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hive.formats.avro;

import com.google.common.base.Verify;
import com.google.common.base.VerifyException;
import com.google.common.primitives.Longs;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.hive.formats.avro.AvroTypeException;
import io.trino.hive.formats.avro.AvroTypeManager;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.SqlTimestamp;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.Type;
import io.trino.spi.type.UuidType;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.apache.avro.LogicalType;
import org.apache.avro.LogicalTypes;
import org.apache.avro.Schema;
import org.apache.avro.SchemaBuilder;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericFixed;

public class NativeLogicalTypesAvroTypeManager
implements AvroTypeManager {
    private static final Logger log = Logger.get(NativeLogicalTypesAvroTypeManager.class);
    public static final Schema TIMESTAMP_MILLIS_SCHEMA = (Schema)SchemaBuilder.builder().longType();
    public static final Schema TIMESTAMP_MICROS_SCHEMA;
    public static final Schema DATE_SCHEMA;
    public static final Schema TIME_MILLIS_SCHEMA;
    public static final Schema TIME_MICROS_SCHEMA;
    public static final Schema UUID_SCHEMA;
    protected static final String DECIMAL = "decimal";
    protected static final String UUID = "uuid";
    protected static final String DATE = "date";
    protected static final String TIME_MILLIS = "time-millis";
    protected static final String TIME_MICROS = "time-micros";
    protected static final String TIMESTAMP_MILLIS = "timestamp-millis";
    protected static final String TIMESTAMP_MICROS = "timestamp-micros";
    protected static final String LOCAL_TIMESTAMP_MILLIS = "local-timestamp-millis";
    protected static final String LOCAL_TIMESTAMP_MICROS = "local-timestamp-micros";
    private static final VarHandle BIG_ENDIAN_LONG_VIEW;

    @Override
    public void configure(Map<String, byte[]> fileMetadata) {
    }

    @Override
    public Optional<Type> overrideTypeForSchema(Schema schema) throws AvroTypeException {
        return this.validateAndLogIssues(schema).map(NativeLogicalTypesAvroTypeManager::getAvroLogicalTypeSpiType);
    }

    @Override
    public Optional<BiConsumer<BlockBuilder, Object>> overrideBuildingFunctionForSchema(Schema schema) throws AvroTypeException {
        return this.validateAndLogIssues(schema).map(logicalType -> NativeLogicalTypesAvroTypeManager.getLogicalTypeBuildingFunction(logicalType, schema));
    }

    @Override
    public Optional<BiFunction<Block, Integer, Object>> overrideBlockToAvroObject(Schema schema, Type type) throws AvroTypeException {
        Optional<LogicalType> logicalType = this.validateAndLogIssues(schema);
        if (logicalType.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(NativeLogicalTypesAvroTypeManager.getAvroFunction(logicalType.get(), schema, type));
    }

    private static Type getAvroLogicalTypeSpiType(LogicalType logicalType) {
        return switch (logicalType.getName()) {
            case TIMESTAMP_MILLIS -> TimestampType.TIMESTAMP_MILLIS;
            case TIMESTAMP_MICROS -> TimestampType.TIMESTAMP_MICROS;
            case DECIMAL -> {
                LogicalTypes.Decimal decimal = (LogicalTypes.Decimal)logicalType;
                yield DecimalType.createDecimalType((int)decimal.getPrecision(), (int)decimal.getScale());
            }
            case DATE -> DateType.DATE;
            case TIME_MILLIS -> TimeType.TIME_MILLIS;
            case TIME_MICROS -> TimeType.TIME_MICROS;
            case UUID -> UuidType.UUID;
            default -> throw new IllegalStateException("Unreachable unfiltered logical type");
        };
    }

    private static BiConsumer<BlockBuilder, Object> getLogicalTypeBuildingFunction(LogicalType logicalType, Schema schema) {
        return switch (logicalType.getName()) {
            case TIMESTAMP_MILLIS -> {
                if (schema.getType() == Schema.Type.LONG) {
                    yield (builder, obj) -> {
                        Long l = (Long)obj;
                        TimestampType.TIMESTAMP_MILLIS.writeLong(builder, l * 1000L);
                    };
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            case TIMESTAMP_MICROS -> {
                if (schema.getType() == Schema.Type.LONG) {
                    yield (builder, obj) -> {
                        Long l = (Long)obj;
                        TimestampType.TIMESTAMP_MICROS.writeLong(builder, l.longValue());
                    };
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            case DECIMAL -> {
                LogicalTypes.Decimal decimal = (LogicalTypes.Decimal)logicalType;
                DecimalType decimalType = DecimalType.createDecimalType((int)decimal.getPrecision(), (int)decimal.getScale());
                Function<Object, byte[]> v1 = switch (schema.getType()) {
                    case Schema.Type.BYTES -> obj -> ((ByteBuffer)obj).array();
                    case Schema.Type.FIXED -> obj -> ((GenericFixed)obj).bytes();
                    default -> throw new IllegalStateException("Unreachable unfiltered logical type");
                };
                Function<Object, byte[]> byteExtract = v1;
                if (decimalType.isShort()) {
                    yield (builder, obj) -> decimalType.writeLong(builder, NativeLogicalTypesAvroTypeManager.fromBigEndian((byte[])byteExtract.apply(obj)));
                }
                yield (builder, obj) -> decimalType.writeObject(builder, (Object)Int128.fromBigEndian((byte[])((byte[])byteExtract.apply(obj))));
            }
            case DATE -> {
                if (schema.getType() == Schema.Type.INT) {
                    yield (builder, obj) -> {
                        Integer i = (Integer)obj;
                        DateType.DATE.writeLong(builder, i.longValue());
                    };
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            case TIME_MILLIS -> {
                if (schema.getType() == Schema.Type.INT) {
                    yield (builder, obj) -> {
                        Integer i = (Integer)obj;
                        TimeType.TIME_MILLIS.writeLong(builder, i.longValue() * 1000000000L);
                    };
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            case TIME_MICROS -> {
                if (schema.getType() == Schema.Type.LONG) {
                    yield (builder, obj) -> {
                        Long i = (Long)obj;
                        TimeType.TIME_MICROS.writeLong(builder, i * 1000000L);
                    };
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            case UUID -> {
                if (schema.getType() == Schema.Type.STRING) {
                    yield (builder, obj) -> UuidType.UUID.writeSlice(builder, UuidType.javaUuidToTrinoUuid((UUID)java.util.UUID.fromString(obj.toString())));
                }
                throw new IllegalStateException("Unreachable unfiltered logical type");
            }
            default -> throw new IllegalStateException("Unreachable unfiltered logical type");
        };
    }

    private static BiFunction<Block, Integer, Object> getAvroFunction(LogicalType logicalType, Schema schema, Type type) throws AvroTypeException {
        return switch (logicalType.getName()) {
            case TIMESTAMP_MILLIS -> {
                if (!(type instanceof TimestampType)) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                TimestampType timestampType = (TimestampType)type;
                if (timestampType.isShort()) {
                    yield (block, integer) -> timestampType.getLong(block, integer.intValue()) / 1000L;
                }
                yield (block, integer) -> {
                    SqlTimestamp timestamp = (SqlTimestamp)timestampType.getObject(block, integer.intValue());
                    return timestamp.roundTo(3).getMillis();
                };
            }
            case TIMESTAMP_MICROS -> {
                if (!(type instanceof TimestampType)) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                TimestampType timestampType = (TimestampType)type;
                if (timestampType.isShort()) {
                    yield (block, position) -> timestampType.getLong(block, position.intValue());
                }
                yield (block, position) -> {
                    SqlTimestamp timestamp = (SqlTimestamp)timestampType.getObject(block, position.intValue());
                    return timestamp.roundTo(6).getEpochMicros();
                };
            }
            case DECIMAL -> {
                DecimalType decimalType = (DecimalType)NativeLogicalTypesAvroTypeManager.getAvroLogicalTypeSpiType(logicalType);
                Function<byte[], Object> v1 = switch (schema.getType()) {
                    case Schema.Type.BYTES -> ByteBuffer::wrap;
                    case Schema.Type.FIXED -> bytes -> new GenericData.Fixed(schema, NativeLogicalTypesAvroTypeManager.fitBigEndianValueToByteArraySize(bytes, schema.getFixedSize()));
                    default -> throw new VerifyException("Unreachable unfiltered logical type");
                };
                Function<byte[], Object> wrapBytes = v1;
                if (decimalType.isShort()) {
                    yield (block, pos) -> wrapBytes.apply(Longs.toByteArray((long)decimalType.getLong(block, pos.intValue())));
                }
                yield (block, pos) -> wrapBytes.apply(((Int128)decimalType.getObject(block, pos.intValue())).toBigEndianBytes());
            }
            case DATE -> {
                if (type != DateType.DATE) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                yield (arg_0, arg_1) -> ((DateType)DateType.DATE).getLong(arg_0, arg_1);
            }
            case TIME_MILLIS -> {
                if (!(type instanceof TimeType)) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                TimeType timeType = (TimeType)type;
                if (timeType.getPrecision() > 3) {
                    throw new AvroTypeException("Can't write out Avro logical time-millis from Trino Time Type with precision %s".formatted(timeType.getPrecision()));
                }
                yield (block, pos) -> Timestamps.roundDiv((long)timeType.getLong(block, pos.intValue()), (long)1000000000L);
            }
            case TIME_MICROS -> {
                if (!(type instanceof TimeType)) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                TimeType timeType = (TimeType)type;
                if (timeType.getPrecision() > 6) {
                    throw new AvroTypeException("Can't write out Avro logical time-millis from Trino Time Type with precision %s".formatted(timeType.getPrecision()));
                }
                yield (block, pos) -> Timestamps.roundDiv((long)timeType.getLong(block, pos.intValue()), (long)1000000L);
            }
            case UUID -> {
                if (!(type instanceof UuidType)) {
                    throw new AvroTypeException("Can't represent Avro logical type %s with Trino Type %s".formatted(logicalType.getName(), type));
                }
                UuidType uuidType = (UuidType)type;
                yield (block, pos) -> UuidType.trinoUuidToJavaUuid((Slice)((Slice)uuidType.getObject(block, pos.intValue()))).toString();
            }
            default -> throw new VerifyException("Unreachable unfiltered logical type");
        };
    }

    private Optional<LogicalType> validateAndLogIssues(Schema schema) {
        ValidateLogicalTypeResult logicalTypeResult = NativeLogicalTypesAvroTypeManager.validateLogicalType(schema);
        if (logicalTypeResult instanceof NoLogicalType) {
            NoLogicalType ignored = (NoLogicalType)logicalTypeResult;
            return Optional.empty();
        }
        if (logicalTypeResult instanceof NonNativeAvroLogicalType) {
            NonNativeAvroLogicalType ignored = (NonNativeAvroLogicalType)logicalTypeResult;
            log.debug("Unrecognized logical type " + String.valueOf(schema));
            return Optional.empty();
        }
        if (logicalTypeResult instanceof InvalidNativeAvroLogicalType) {
            InvalidNativeAvroLogicalType invalidNativeAvroLogicalType = (InvalidNativeAvroLogicalType)logicalTypeResult;
            log.debug((Throwable)invalidNativeAvroLogicalType.getCause(), "Invalidly configured native Avro logical type");
            return Optional.empty();
        }
        if (logicalTypeResult instanceof ValidNativeAvroLogicalType) {
            ValidNativeAvroLogicalType validNativeAvroLogicalType = (ValidNativeAvroLogicalType)logicalTypeResult;
            return Optional.of(validNativeAvroLogicalType.getLogicalType());
        }
        throw new IllegalStateException("Unhandled validate logical type result");
    }

    protected static ValidateLogicalTypeResult validateLogicalType(Schema schema) {
        LogicalType logicalType;
        String typeName = schema.getProp("logicalType");
        if (typeName == null) {
            return new NoLogicalType();
        }
        switch (typeName) {
            case "timestamp-millis": 
            case "timestamp-micros": 
            case "decimal": 
            case "date": 
            case "time-millis": 
            case "time-micros": 
            case "uuid": {
                logicalType = LogicalTypes.fromSchemaIgnoreInvalid((Schema)schema);
                break;
            }
            case "local-timestamp-microslocal-timestamp-millis": {
                log.debug("Logical type " + typeName + " not currently supported by by Trino");
            }
            default: {
                return new NonNativeAvroLogicalType(typeName);
            }
        }
        if (logicalType != null) {
            try {
                logicalType.validate(schema);
            }
            catch (RuntimeException e) {
                return new InvalidNativeAvroLogicalType(typeName, e);
            }
            return new ValidNativeAvroLogicalType(logicalType);
        }
        return new NonNativeAvroLogicalType(typeName);
    }

    public static long fromBigEndian(byte[] bytes) {
        if (bytes.length > 8) {
            int offset = bytes.length - 8;
            long res = BIG_ENDIAN_LONG_VIEW.get(bytes, offset);
            int expectedSignExtensionByte = (int)(res >> 63);
            for (int i = 0; i < offset; ++i) {
                if (bytes[i] == expectedSignExtensionByte) continue;
                throw new ArithmeticException("Overflow");
            }
            return res;
        }
        if (bytes.length == 8) {
            return BIG_ENDIAN_LONG_VIEW.get(bytes, 0);
        }
        long res = bytes[0] >> 7;
        for (byte b : bytes) {
            res = res << 8 | (long)(b & 0xFF);
        }
        return res;
    }

    public static byte[] fitBigEndianValueToByteArraySize(long value, int byteSize) {
        return NativeLogicalTypesAvroTypeManager.fitBigEndianValueToByteArraySize(Longs.toByteArray((long)value), byteSize);
    }

    public static byte[] fitBigEndianValueToByteArraySize(Int128 value, int byteSize) {
        return NativeLogicalTypesAvroTypeManager.fitBigEndianValueToByteArraySize(value.toBigEndianBytes(), byteSize);
    }

    public static byte[] fitBigEndianValueToByteArraySize(byte[] value, int byteSize) {
        if (value.length == byteSize) {
            return value;
        }
        if (value.length < byteSize) {
            return NativeLogicalTypesAvroTypeManager.padBigEndianToSize(value, byteSize);
        }
        if (NativeLogicalTypesAvroTypeManager.canBigEndianValueBeRepresentedBySmallerByteSize(value, byteSize)) {
            byte[] dest = new byte[byteSize];
            System.arraycopy(value, value.length - byteSize, dest, 0, byteSize);
            return dest;
        }
        throw new ArithmeticException("Can't resize big endian bytes %s to size %s".formatted(Arrays.toString(value), byteSize));
    }

    private static boolean canBigEndianValueBeRepresentedBySmallerByteSize(byte[] bigEndianValue, int byteSize) {
        Verify.verify((byteSize < bigEndianValue.length ? 1 : 0) != 0);
        if (byteSize <= 0) {
            return false;
        }
        if (bigEndianValue[0] != 0 && bigEndianValue[0] != -1) {
            return false;
        }
        int firstSigByte = 0;
        byte padding = bigEndianValue[0];
        for (int i = 1; i < bigEndianValue.length; ++i) {
            if (bigEndianValue[i] == padding) {
                firstSigByte = i;
                continue;
            }
            if (padding == 0 && bigEndianValue[i] < 0) break;
            if (padding == 0 && bigEndianValue[i] > 0) {
                firstSigByte = i;
                break;
            }
            if (padding == -1 && bigEndianValue[i] >= 0) break;
            if (padding != -1 || bigEndianValue[i] >= 0) continue;
            firstSigByte = i;
            break;
        }
        return bigEndianValue.length - firstSigByte <= byteSize;
    }

    public static byte[] padBigEndianToSize(Int128 toPad, int byteSize) {
        return NativeLogicalTypesAvroTypeManager.padBigEndianToSize(toPad.toBigEndianBytes(), byteSize);
    }

    public static byte[] padBigEndianToSize(long toPad, int byteSize) {
        return NativeLogicalTypesAvroTypeManager.padBigEndianToSize(Longs.toByteArray((long)toPad), byteSize);
    }

    public static byte[] padBigEndianToSize(byte[] toPad, int byteSize) {
        int endianSize = toPad.length;
        if (byteSize < endianSize) {
            throw new ArithmeticException("Big endian bytes size must be less than or equal to the total padded size");
        }
        if (endianSize < 1) {
            throw new ArithmeticException("Cannot pad empty array");
        }
        byte[] padded = new byte[byteSize];
        System.arraycopy(toPad, 0, padded, byteSize - endianSize, endianSize);
        if (toPad[0] < 0) {
            for (int i = 0; i < byteSize - endianSize; ++i) {
                padded[i] = -1;
            }
        }
        return padded;
    }

    static {
        LogicalTypes.timestampMillis().addToSchema(TIMESTAMP_MILLIS_SCHEMA);
        TIMESTAMP_MICROS_SCHEMA = (Schema)SchemaBuilder.builder().longType();
        LogicalTypes.timestampMicros().addToSchema(TIMESTAMP_MICROS_SCHEMA);
        DATE_SCHEMA = Schema.create((Schema.Type)Schema.Type.INT);
        LogicalTypes.date().addToSchema(DATE_SCHEMA);
        TIME_MILLIS_SCHEMA = Schema.create((Schema.Type)Schema.Type.INT);
        LogicalTypes.timeMillis().addToSchema(TIME_MILLIS_SCHEMA);
        TIME_MICROS_SCHEMA = Schema.create((Schema.Type)Schema.Type.LONG);
        LogicalTypes.timeMicros().addToSchema(TIME_MICROS_SCHEMA);
        UUID_SCHEMA = Schema.create((Schema.Type)Schema.Type.STRING);
        LogicalTypes.uuid().addToSchema(UUID_SCHEMA);
        BIG_ENDIAN_LONG_VIEW = MethodHandles.byteArrayViewVarHandle(long[].class, ByteOrder.BIG_ENDIAN);
    }

    protected static abstract sealed class ValidateLogicalTypeResult
    permits NoLogicalType, NonNativeAvroLogicalType, InvalidNativeAvroLogicalType, ValidNativeAvroLogicalType {
        protected ValidateLogicalTypeResult() {
        }
    }

    protected static final class NoLogicalType
    extends ValidateLogicalTypeResult {
        protected NoLogicalType() {
        }
    }

    protected static final class NonNativeAvroLogicalType
    extends ValidateLogicalTypeResult {
        private final String logicalTypeName;

        public NonNativeAvroLogicalType(String logicalTypeName) {
            this.logicalTypeName = Objects.requireNonNull(logicalTypeName, "logicalTypeName is null");
        }

        public String getLogicalTypeName() {
            return this.logicalTypeName;
        }
    }

    protected static final class InvalidNativeAvroLogicalType
    extends ValidateLogicalTypeResult {
        private final String logicalTypeName;
        private final RuntimeException cause;

        public InvalidNativeAvroLogicalType(String logicalTypeName, RuntimeException cause) {
            this.logicalTypeName = Objects.requireNonNull(logicalTypeName, "logicalTypeName");
            this.cause = Objects.requireNonNull(cause, "cause is null");
        }

        public String getLogicalTypeName() {
            return this.logicalTypeName;
        }

        public RuntimeException getCause() {
            return this.cause;
        }
    }

    protected static final class ValidNativeAvroLogicalType
    extends ValidateLogicalTypeResult {
        private final LogicalType logicalType;

        public ValidNativeAvroLogicalType(LogicalType logicalType) {
            this.logicalType = Objects.requireNonNull(logicalType, "logicalType is null");
        }

        public LogicalType getLogicalType() {
            return this.logicalType;
        }
    }
}

