/*
 * Decompiled with CFR 0.152.
 */
package io.trino.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.airlift.log.Logger;
import io.trino.hdfs.FileSystemFinalizerService;
import io.trino.hdfs.TrinoFileSystemCacheStats;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemCache;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.UserGroupInformationShim;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;
import org.gaul.modernizer_maven_annotations.SuppressModernizer;

public class TrinoFileSystemCache
implements FileSystemCache {
    private static final Logger log = Logger.get(TrinoFileSystemCache.class);
    public static final String CACHE_KEY = "fs.cache.credentials";
    public static final TrinoFileSystemCache INSTANCE = new TrinoFileSystemCache();
    private final AtomicLong unique = new AtomicLong();
    private final TrinoFileSystemCacheStats stats;
    @GuardedBy(value="this")
    private final Map<FileSystemKey, FileSystemHolder> map = new HashMap<FileSystemKey, FileSystemHolder>();

    @VisibleForTesting
    TrinoFileSystemCache() {
        this.stats = new TrinoFileSystemCacheStats(() -> {
            TrinoFileSystemCache trinoFileSystemCache = this;
            synchronized (trinoFileSystemCache) {
                return this.map.size();
            }
        });
    }

    public FileSystem get(URI uri, Configuration conf) throws IOException {
        this.stats.newGetCall();
        return this.getInternal(uri, conf, 0L);
    }

    public FileSystem getUnique(URI uri, Configuration conf) throws IOException {
        this.stats.newGetUniqueCall();
        return this.getInternal(uri, conf, this.unique.incrementAndGet());
    }

    @VisibleForTesting
    int getCacheSize() {
        return this.map.size();
    }

    private synchronized FileSystem getInternal(URI uri, Configuration conf, long unique) throws IOException {
        UserGroupInformation userGroupInformation = UserGroupInformation.getCurrentUser();
        FileSystemKey key = TrinoFileSystemCache.createFileSystemKey(uri, userGroupInformation, unique);
        Set<?> privateCredentials = TrinoFileSystemCache.getPrivateCredentials(userGroupInformation);
        FileSystemHolder fileSystemHolder = this.map.get(key);
        if (fileSystemHolder == null) {
            int maxSize = conf.getInt("fs.cache.max-size", 1000);
            if (this.map.size() >= maxSize) {
                this.stats.newGetCallFailed();
                throw new IOException(String.format("FileSystem max cache size has been reached: %s", maxSize));
            }
            try {
                FileSystem fileSystem = TrinoFileSystemCache.createFileSystem(uri, conf);
                fileSystemHolder = new FileSystemHolder(fileSystem, privateCredentials);
                this.map.put(key, fileSystemHolder);
            }
            catch (IOException e) {
                this.stats.newGetCallFailed();
                throw e;
            }
        }
        if (TrinoFileSystemCache.isHdfs(uri) && !fileSystemHolder.getPrivateCredentials().equals(privateCredentials) || TrinoFileSystemCache.extraCredentialsChanged(fileSystemHolder.getFileSystem(), conf)) {
            this.map.remove(key);
            try {
                FileSystem fileSystem = TrinoFileSystemCache.createFileSystem(uri, conf);
                fileSystemHolder = new FileSystemHolder(fileSystem, privateCredentials);
                this.map.put(key, fileSystemHolder);
            }
            catch (IOException e) {
                this.stats.newGetCallFailed();
                throw e;
            }
        }
        return fileSystemHolder.getFileSystem();
    }

    private static FileSystem createFileSystem(URI uri, Configuration conf) throws IOException {
        Class clazz = FileSystem.getFileSystemClass((String)uri.getScheme(), (Configuration)conf);
        if (clazz == null) {
            throw new IOException("No FileSystem for scheme: " + uri.getScheme());
        }
        FileSystem original = (FileSystem)ReflectionUtils.newInstance((Class)clazz, (Configuration)conf);
        original.initialize(uri, conf);
        FileSystemWrapper wrapper = new FileSystemWrapper(original);
        FileSystemFinalizerService.getInstance().addFinalizer((Object)wrapper, () -> {
            try {
                TrinoFileSystemCache.closeFileSystem(original);
            }
            catch (IOException e) {
                log.error((Throwable)e, "Error occurred when finalizing file system");
            }
        });
        return wrapper;
    }

    public synchronized void remove(FileSystem fileSystem) {
        this.stats.newRemoveCall();
        this.map.values().removeIf(holder -> holder.getFileSystem().equals(fileSystem));
    }

    public synchronized void closeAll() throws IOException {
        for (FileSystemHolder fileSystemHolder : ImmutableList.copyOf(this.map.values())) {
            TrinoFileSystemCache.closeFileSystem(fileSystemHolder.getFileSystem());
        }
        this.map.clear();
    }

    @SuppressModernizer
    private static void closeFileSystem(FileSystem fileSystem) throws IOException {
        fileSystem.close();
    }

    private static FileSystemKey createFileSystemKey(URI uri, UserGroupInformation userGroupInformation, long unique) {
        String realUser;
        String scheme = Strings.nullToEmpty((String)uri.getScheme()).toLowerCase(Locale.ENGLISH);
        String authority = Strings.nullToEmpty((String)uri.getAuthority()).toLowerCase(Locale.ENGLISH);
        UserGroupInformation.AuthenticationMethod authenticationMethod = userGroupInformation.getAuthenticationMethod();
        return new FileSystemKey(scheme, authority, unique, realUser, switch (authenticationMethod) {
            case UserGroupInformation.AuthenticationMethod.SIMPLE, UserGroupInformation.AuthenticationMethod.KERBEROS -> {
                realUser = userGroupInformation.getUserName();
                yield null;
            }
            case UserGroupInformation.AuthenticationMethod.PROXY -> {
                realUser = userGroupInformation.getRealUser().getUserName();
                yield userGroupInformation.getUserName();
            }
            default -> throw new IllegalArgumentException("Unsupported authentication method: " + authenticationMethod);
        });
    }

    private static Set<?> getPrivateCredentials(UserGroupInformation userGroupInformation) {
        UserGroupInformation.AuthenticationMethod authenticationMethod = userGroupInformation.getAuthenticationMethod();
        switch (authenticationMethod) {
            case SIMPLE: {
                return ImmutableSet.of();
            }
            case KERBEROS: {
                return ImmutableSet.copyOf(UserGroupInformationShim.getSubject((UserGroupInformation)userGroupInformation).getPrivateCredentials());
            }
            case PROXY: {
                return TrinoFileSystemCache.getPrivateCredentials(userGroupInformation.getRealUser());
            }
        }
        throw new IllegalArgumentException("Unsupported authentication method: " + authenticationMethod);
    }

    private static boolean isHdfs(URI uri) {
        String scheme = uri.getScheme();
        return "hdfs".equals(scheme) || "viewfs".equals(scheme);
    }

    private static boolean extraCredentialsChanged(FileSystem fileSystem, Configuration configuration) {
        return !configuration.get(CACHE_KEY, "").equals(fileSystem.getConf().get(CACHE_KEY, ""));
    }

    public TrinoFileSystemCacheStats getFileSystemCacheStats() {
        return this.stats;
    }

    private static class FileSystemKey {
        private final String scheme;
        private final String authority;
        private final long unique;
        private final String realUser;
        private final String proxyUser;

        public FileSystemKey(String scheme, String authority, long unique, String realUser, String proxyUser) {
            this.scheme = Objects.requireNonNull(scheme, "scheme is null");
            this.authority = Objects.requireNonNull(authority, "authority is null");
            this.unique = unique;
            this.realUser = Objects.requireNonNull(realUser, "realUser");
            this.proxyUser = proxyUser;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FileSystemKey that = (FileSystemKey)o;
            return Objects.equals(this.scheme, that.scheme) && Objects.equals(this.authority, that.authority) && Objects.equals(this.unique, that.unique) && Objects.equals(this.realUser, that.realUser) && Objects.equals(this.proxyUser, that.proxyUser);
        }

        public int hashCode() {
            return Objects.hash(this.scheme, this.authority, this.unique, this.realUser, this.proxyUser);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("scheme", (Object)this.scheme).add("authority", (Object)this.authority).add("unique", this.unique).add("realUser", (Object)this.realUser).add("proxyUser", (Object)this.proxyUser).toString();
        }
    }

    private static class FileSystemHolder {
        private final FileSystem fileSystem;
        private final Set<?> privateCredentials;

        public FileSystemHolder(FileSystem fileSystem, Set<?> privateCredentials) {
            this.fileSystem = Objects.requireNonNull(fileSystem, "fileSystem is null");
            this.privateCredentials = ImmutableSet.copyOf((Collection)Objects.requireNonNull(privateCredentials, "privateCredentials is null"));
        }

        public FileSystem getFileSystem() {
            return this.fileSystem;
        }

        public Set<?> getPrivateCredentials() {
            return this.privateCredentials;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("fileSystem", (Object)this.fileSystem).add("privateCredentials", this.privateCredentials).toString();
        }
    }

    private static class FileSystemWrapper
    extends FilterFileSystem {
        public FileSystemWrapper(FileSystem fs) {
            super(fs);
        }

        public FSDataInputStream open(Path f, int bufferSize) throws IOException {
            return new InputStreamWrapper(this.getRawFileSystem().open(f, bufferSize), (FileSystem)this);
        }

        public String getScheme() {
            return this.getRawFileSystem().getScheme();
        }

        public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().append(f, bufferSize, progress), (FileSystem)this);
        }

        public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().create(f, permission, overwrite, bufferSize, replication, blockSize, progress), (FileSystem)this);
        }

        public FSDataOutputStream create(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress, Options.ChecksumOpt checksumOpt) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().create(f, permission, flags, bufferSize, replication, blockSize, progress, checksumOpt), (FileSystem)this);
        }

        public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            return new OutputStreamWrapper(this.getRawFileSystem().createNonRecursive(f, permission, flags, bufferSize, replication, blockSize, progress), (FileSystem)this);
        }

        public BlockLocation[] getFileBlockLocations(Path p, long start, long len) throws IOException {
            return this.fs.getFileBlockLocations(p, start, len);
        }

        public RemoteIterator<LocatedFileStatus> listFiles(Path path, boolean recursive) throws IOException {
            return this.fs.listFiles(path, recursive);
        }
    }

    private static class InputStreamWrapper
    extends FSDataInputStream {
        private final FileSystem fileSystem;

        public InputStreamWrapper(FSDataInputStream inputStream, FileSystem fileSystem) {
            super((InputStream)inputStream);
            this.fileSystem = fileSystem;
        }

        public InputStream getWrappedStream() {
            return ((FSDataInputStream)super.getWrappedStream()).getWrappedStream();
        }
    }

    private static class OutputStreamWrapper
    extends FSDataOutputStream {
        private final FileSystem fileSystem;

        public OutputStreamWrapper(FSDataOutputStream delegate, FileSystem fileSystem) {
            super((OutputStream)delegate, null, delegate.getPos());
            this.fileSystem = fileSystem;
        }

        public OutputStream getWrappedStream() {
            return ((FSDataOutputStream)super.getWrappedStream()).getWrappedStream();
        }
    }
}

