/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.exec.spark.session;

import io.trino.hive.$internal.com.google.common.annotations.VisibleForTesting;
import io.trino.hive.$internal.com.google.common.base.Preconditions;
import io.trino.hive.$internal.com.google.common.base.Throwables;
import io.trino.hive.$internal.com.google.common.collect.ImmutableMap;
import io.trino.hive.$internal.com.google.common.collect.Maps;
import io.trino.hive.$internal.org.slf4j.Logger;
import io.trino.hive.$internal.org.slf4j.LoggerFactory;
import java.io.IOException;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.common.ObjectPair;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.DriverContext;
import org.apache.hadoop.hive.ql.ErrorMsg;
import org.apache.hadoop.hive.ql.exec.spark.HiveSparkClient;
import org.apache.hadoop.hive.ql.exec.spark.HiveSparkClientFactory;
import org.apache.hadoop.hive.ql.exec.spark.session.SparkSession;
import org.apache.hadoop.hive.ql.exec.spark.status.SparkJobRef;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.plan.SparkWork;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.spark.SparkConf;
import org.apache.spark.util.Utils;

public class SparkSessionImpl
implements SparkSession {
    private static final Logger LOG = LoggerFactory.getLogger(SparkSession.class);
    private static final String SPARK_DIR = "_spark_session_dir";
    private static final String AM_TIMEOUT_ERR = ".*ApplicationMaster for attempt.*timed out.*";
    private static final String UNKNOWN_QUEUE_ERR = "(submitted by user.*to unknown queue:.*)\n";
    private static final String STOPPED_QUEUE_ERR = "(Queue.*is STOPPED)";
    private static final String FULL_QUEUE_ERR = "(Queue.*already has.*applications)";
    private static final String INVALILD_MEM_ERR = "(Required executor memory.*is above the max threshold.*) of this";
    private static final String INVALID_CORE_ERR = "(initial executor number.*must between min executor.*and max executor number.*)\n";
    private static Map<String, Pattern> errorPatterns;
    private HiveConf conf;
    private boolean isOpen;
    private final String sessionId;
    private HiveSparkClient hiveSparkClient;
    private Path scratchDir;
    private final Object dirLock = new Object();
    private String matchedString = null;

    public SparkSessionImpl() {
        this.sessionId = SparkSessionImpl.makeSessionId();
        SparkSessionImpl.initErrorPatterns();
    }

    @Override
    public void open(HiveConf conf) throws HiveException {
        LOG.info("Trying to open Spark session {}", (Object)this.sessionId);
        this.conf = conf;
        this.isOpen = true;
        try {
            this.hiveSparkClient = HiveSparkClientFactory.createHiveSparkClient(conf, this.sessionId);
        }
        catch (Throwable e) {
            HiveException he = this.isOpen ? this.getHiveException(e) : new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_CLOSED_SESSION, this.sessionId);
            throw he;
        }
        LOG.info("Spark session {} is successfully opened", (Object)this.sessionId);
    }

    @Override
    public SparkJobRef submit(DriverContext driverContext, SparkWork sparkWork) throws Exception {
        Preconditions.checkState(this.isOpen, "Session is not open. Can't submit jobs.");
        return this.hiveSparkClient.execute(driverContext, sparkWork);
    }

    @Override
    public ObjectPair<Long, Integer> getMemoryAndCores() throws Exception {
        int totalCores;
        SparkConf sparkConf = this.hiveSparkClient.getSparkConf();
        int numExecutors = this.hiveSparkClient.getExecutorCount();
        if (numExecutors <= 0) {
            return new ObjectPair<Long, Integer>(-1L, -1);
        }
        int executorMemoryInMB = Utils.memoryStringToMb((String)sparkConf.get("spark.executor.memory", "512m"));
        double memoryFraction = 1.0 - sparkConf.getDouble("spark.storage.memoryFraction", 0.6);
        long totalMemory = (long)((double)(numExecutors * executorMemoryInMB) * memoryFraction * 1024.0 * 1024.0);
        String masterURL = sparkConf.get("spark.master");
        if (masterURL.startsWith("spark") || masterURL.startsWith("local")) {
            totalCores = sparkConf.contains("spark.default.parallelism") ? sparkConf.getInt("spark.default.parallelism", 1) : this.hiveSparkClient.getDefaultParallelism();
            totalCores = Math.max(totalCores, numExecutors);
        } else {
            int coresPerExecutor = sparkConf.getInt("spark.executor.cores", 1);
            totalCores = numExecutors * coresPerExecutor;
        }
        long memoryPerTaskInBytes = totalMemory / (long)(totalCores /= sparkConf.getInt("spark.task.cpus", 1));
        LOG.info("Spark cluster current has executors: " + numExecutors + ", total cores: " + totalCores + ", memory per executor: " + executorMemoryInMB + "M, memoryFraction: " + memoryFraction);
        return new ObjectPair<Long, Integer>(memoryPerTaskInBytes, totalCores);
    }

    @Override
    public boolean isOpen() {
        return this.isOpen;
    }

    @Override
    public HiveConf getConf() {
        return this.conf;
    }

    @Override
    public String getSessionId() {
        return this.sessionId;
    }

    @Override
    public void close() {
        LOG.info("Trying to close Spark session {}", (Object)this.sessionId);
        this.isOpen = false;
        if (this.hiveSparkClient != null) {
            try {
                this.hiveSparkClient.close();
                LOG.info("Spark session {} is successfully closed", (Object)this.sessionId);
                this.cleanScratchDir();
            }
            catch (IOException e) {
                LOG.error("Failed to close spark session (" + this.sessionId + ").", e);
            }
        }
        this.hiveSparkClient = null;
    }

    private Path createScratchDir() throws IOException {
        Path parent = new Path(SessionState.get().getHdfsScratchDirURIString(), SPARK_DIR);
        Path sparkDir = new Path(parent, this.sessionId);
        FileSystem fs = sparkDir.getFileSystem((Configuration)this.conf);
        FsPermission fsPermission = new FsPermission(HiveConf.getVar(this.conf, HiveConf.ConfVars.SCRATCHDIRPERMISSION));
        fs.mkdirs(sparkDir, fsPermission);
        fs.deleteOnExit(sparkDir);
        return sparkDir;
    }

    private static void initErrorPatterns() {
        errorPatterns = Maps.newHashMap(new ImmutableMap.Builder<String, Pattern>().put(AM_TIMEOUT_ERR, Pattern.compile(AM_TIMEOUT_ERR)).put(UNKNOWN_QUEUE_ERR, Pattern.compile(UNKNOWN_QUEUE_ERR)).put(STOPPED_QUEUE_ERR, Pattern.compile(STOPPED_QUEUE_ERR)).put(FULL_QUEUE_ERR, Pattern.compile(FULL_QUEUE_ERR)).put(INVALILD_MEM_ERR, Pattern.compile(INVALILD_MEM_ERR)).put(INVALID_CORE_ERR, Pattern.compile(INVALID_CORE_ERR)).build());
    }

    @VisibleForTesting
    HiveException getHiveException(Throwable e) {
        Throwable oe = e;
        while (e != null) {
            if (e instanceof TimeoutException) {
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_TIMEOUT);
            }
            if (e instanceof InterruptedException) {
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INTERRUPTED, this.sessionId);
            }
            if (e instanceof RuntimeException) {
                String sts = Throwables.getStackTraceAsString(e);
                if (this.matches(sts, AM_TIMEOUT_ERR)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_TIMEOUT);
                }
                if (this.matches(sts, UNKNOWN_QUEUE_ERR) || this.matches(sts, STOPPED_QUEUE_ERR)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INVALID_QUEUE, this.matchedString);
                }
                if (this.matches(sts, FULL_QUEUE_ERR)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_QUEUE_FULL, this.matchedString);
                }
                if (this.matches(sts, INVALILD_MEM_ERR) || this.matches(sts, INVALID_CORE_ERR)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INVALID_RESOURCE_REQUEST, this.matchedString);
                }
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_ERROR, this.sessionId);
            }
            e = e.getCause();
        }
        return new HiveException(oe, ErrorMsg.SPARK_CREATE_CLIENT_ERROR, this.sessionId);
    }

    @VisibleForTesting
    String getMatchedString() {
        return this.matchedString;
    }

    private boolean matches(String input, String regex) {
        if (!errorPatterns.containsKey(regex)) {
            LOG.warn("No error pattern found for regex: {}", (Object)regex);
            return false;
        }
        Pattern p = errorPatterns.get(regex);
        Matcher m = p.matcher(input);
        boolean result = m.find();
        if (result && m.groupCount() == 1) {
            this.matchedString = m.group(1);
        }
        return result;
    }

    private void cleanScratchDir() throws IOException {
        if (this.scratchDir != null) {
            FileSystem fs = this.scratchDir.getFileSystem((Configuration)this.conf);
            fs.delete(this.scratchDir, true);
            this.scratchDir = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Path getHDFSSessionDir() throws IOException {
        if (this.scratchDir == null) {
            Object object = this.dirLock;
            synchronized (object) {
                if (this.scratchDir == null) {
                    this.scratchDir = this.createScratchDir();
                }
            }
        }
        return this.scratchDir;
    }

    public static String makeSessionId() {
        return UUID.randomUUID().toString();
    }

    @VisibleForTesting
    HiveSparkClient getHiveSparkClient() {
        return this.hiveSparkClient;
    }
}

