package io.vertx.mutiny.core.http;

import java.util.Map;
import java.util.stream.Collectors;
import io.smallrye.mutiny.Multi;
import io.smallrye.mutiny.Uni;
import io.smallrye.mutiny.vertx.TypeArg;
import io.vertx.codegen.annotations.Fluent;
import io.smallrye.common.annotation.CheckReturnValue;
import io.vertx.core.http.StreamPriority;
import io.vertx.core.Future;
import java.util.function.Function;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;

/**
 * Represents a client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.mutiny.core.http.HttpClient} instance, via one of the methods corresponding to the
 * specific HTTP methods, or the generic request methods. On creation the request will not have been written to the
 * wire.
 * <p>
 * Once a request has been obtained, headers can be set on it, and data can be written to its body if required. Once
 * you are ready to send the request, one of the {@link io.vertx.mutiny.core.http.HttpClientRequest#end} methods should be called.
 * <p>
 * Nothing is actually sent until the request has been internally assigned an HTTP connection.
 * <p>
 * The {@link io.vertx.mutiny.core.http.HttpClient} instance will return an instance of this class immediately, even if there are no HTTP
 * connections available in the pool. Any requests sent before a connection is assigned will be queued
 * internally and actually sent when an HTTP connection becomes available from the pool.
 * <p>
 * The headers of the request are queued for writing either when the {@link io.vertx.mutiny.core.http.HttpClientRequest#end} method is called, or, when the first
 * part of the body is written, whichever occurs first.
 * <p>
 * This class supports both chunked and non-chunked HTTP.
 * <p>
 * It implements {@link io.vertx.mutiny.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.mutiny.core.streams.Pipe} to pipe data with flow control.
 * <p>
 * An example of using this class is as follows:
 * <p>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClientRequest original} non Mutiny-ified interface using Vert.x codegen.
 */

@io.smallrye.mutiny.vertx.MutinyGen(io.vertx.core.http.HttpClientRequest.class)
public class HttpClientRequest implements io.smallrye.mutiny.vertx.MutinyDelegate, io.vertx.mutiny.core.streams.WriteStream<io.vertx.mutiny.core.buffer.Buffer> {

  public static final io.smallrye.mutiny.vertx.TypeArg<HttpClientRequest> __TYPE_ARG = new io.smallrye.mutiny.vertx.TypeArg<>(    obj -> new HttpClientRequest((io.vertx.core.http.HttpClientRequest) obj),
    HttpClientRequest::getDelegate
  );

  private final io.vertx.core.http.HttpClientRequest delegate;
  
  public HttpClientRequest(io.vertx.core.http.HttpClientRequest delegate) {
    this.delegate = delegate;
  }

  public HttpClientRequest(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpClientRequest)delegate;
  }

  /**
   * Empty constructor used by CDI, do not use this constructor directly.
   **/
  HttpClientRequest() {
    this.delegate = null;
  }

  @Override
  public io.vertx.core.http.HttpClientRequest getDelegate() {
    return delegate;
  }

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClientRequest that = (HttpClientRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  private io.smallrye.mutiny.vertx.WriteStreamSubscriber<io.vertx.mutiny.core.buffer.Buffer> subscriber;

  @CheckReturnValue
  public synchronized io.smallrye.mutiny.vertx.WriteStreamSubscriber<io.vertx.mutiny.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      java.util.function.Function<io.vertx.mutiny.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.mutiny.core.buffer.Buffer::getDelegate;
      subscriber = io.smallrye.mutiny.vertx.MutinyHelper.toSubscriber(getDelegate(), conv);
    }
    return subscriber;
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param data 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> write(io.vertx.mutiny.core.buffer.Buffer data) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.write(data.getDelegate(), handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.streams.WriteStream#write(io.vertx.mutiny.core.buffer.Buffer)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param data 
   * @return the Void instance produced by the operation.
   */
  public Void writeAndAwait(io.vertx.mutiny.core.buffer.Buffer data) { 
    return (Void) write(data).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.streams.WriteStream#write(io.vertx.mutiny.core.buffer.Buffer)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.streams.WriteStream#write(io.vertx.mutiny.core.buffer.Buffer)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.streams.WriteStream#write(io.vertx.mutiny.core.buffer.Buffer)} but you don't need to compose it with other operations.
   * @param data 
   */
  public void writeAndForget(io.vertx.mutiny.core.buffer.Buffer data) { 
    write(data).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * @return <code>true</code> if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  @Fluent
  private io.vertx.mutiny.core.http.HttpClientRequest __exceptionHandler(Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.mutiny.core.http.HttpClientRequest exceptionHandler(java.util.function.Consumer<java.lang.Throwable> handler) {
    return __exceptionHandler(io.smallrye.mutiny.vertx.MutinyHelper.convertConsumer(handler));
  }

  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  @Fluent
  private io.vertx.mutiny.core.http.HttpClientRequest __drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  public io.vertx.mutiny.core.http.HttpClientRequest drainHandler(java.lang.Runnable handler) {
    return __drainHandler(ignored -> handler.run()
);
  }

  /**
   * @param authority override the request authority
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest authority(io.vertx.mutiny.core.net.HostAndPort authority) { 
    delegate.authority(authority.getDelegate());
    return this;
  }

  /**
   * @param host the host part of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @return a reference to this, so the API can be used fluently
   * @deprecated instead use {@link #authority(HostAndPort)}   */
  @Deprecated
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setHost(String host) { 
    delegate.setHost(host);
    return this;
  }

  /**
   * @return the host value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @deprecated   */
  @Deprecated
  public String getHost() { 
    String ret = delegate.getHost();
    return ret;
  }

  /**
   * @param port the port part of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @return a reference to this, so the API can be used fluently
   * @deprecated instead use {@link #authority(HostAndPort)}   */
  @Deprecated
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setPort(int port) { 
    delegate.setPort(port);
    return this;
  }

  /**
   * @return the port value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @deprecated   */
  @Deprecated
  public int getPort() { 
    int ret = delegate.getPort();
    return ret;
  }

  /**
   * @param followRedirects <code>true</code> to follow HTTP redirects
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setFollowRedirects(boolean followRedirects) { 
    delegate.setFollowRedirects(followRedirects);
    return this;
  }

  /**
   * @return whether HTTP redirections should be followed
   */
  public boolean isFollowRedirects() { 
    boolean ret = delegate.isFollowRedirects();
    return ret;
  }

  /**
   * @param maxRedirects the number of HTTP redirect to follow
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setMaxRedirects(int maxRedirects) { 
    delegate.setMaxRedirects(maxRedirects);
    return this;
  }

  /**
   * @return the maximum number of HTTP redirections to follow
   */
  public int getMaxRedirects() { 
    int ret = delegate.getMaxRedirects();
    return ret;
  }

  /**
   * @return the number of followed redirections for the current HTTP request
   */
  public int numberOfRedirections() { 
    int ret = delegate.numberOfRedirections();
    return ret;
  }

  /**
   * @param chunked true if chunked encoding
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return Is the request chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * @return 
   */
  public io.vertx.core.http.HttpMethod getMethod() { 
    io.vertx.core.http.HttpMethod ret = delegate.getMethod();
    return ret;
  }

  /**
   * @param method the HTTP method
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setMethod(io.vertx.core.http.HttpMethod method) { 
    delegate.setMethod(method);
    return this;
  }

  /**
   * @return the absolute URI corresponding to the HTTP request
   */
  public String absoluteURI() { 
    String ret = delegate.absoluteURI();
    return ret;
  }

  /**
   * @return The URI of the request.
   */
  public String getURI() { 
    String ret = delegate.getURI();
    return ret;
  }

  /**
   * @param uri the request uri
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setURI(String uri) { 
    delegate.setURI(uri);
    return this;
  }

  /**
   * @return The path part of the uri. For example /somepath/somemorepath/someresource.foo
   */
  public String path() { 
    String ret = delegate.path();
    return ret;
  }

  /**
   * @return the query part of the uri. For example someparam=32&amp;someotherparam=x
   */
  public String query() { 
    String ret = delegate.query();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.mutiny.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.mutiny.core.MultiMap ret = io.vertx.mutiny.core.MultiMap.newInstance((io.vertx.core.MultiMap)delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * @param name The header name
   * @param value The header value
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @param op the operation
   * @return @return a reference to this, so the API can be used fluently
   */
  public io.vertx.mutiny.core.http.HttpClientRequest traceOperation(String op) { 
    io.vertx.mutiny.core.http.HttpClientRequest ret = io.vertx.mutiny.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)delegate.traceOperation(op));
    return ret;
  }

  /**
   * @return the trace operation of this request
   */
  public String traceOperation() { 
    String ret = delegate.traceOperation();
    return ret;
  }

  /**
   * @return the HTTP version for this request
   */
  public io.vertx.core.http.HttpVersion version() { 
    io.vertx.core.http.HttpVersion ret = delegate.version();
    return ret;
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param chunk 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> write(String chunk) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.write(chunk, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param chunk 
   * @return the Void instance produced by the operation.
   */
  public Void writeAndAwait(String chunk) { 
    return (Void) write(chunk).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String)} but you don't need to compose it with other operations.
   * @param chunk 
   */
  public void writeAndForget(String chunk) { 
    write(chunk).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param chunk 
   * @param enc 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> write(String chunk, String enc) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.write(chunk, enc, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param chunk 
   * @param enc 
   * @return the Void instance produced by the operation.
   */
  public Void writeAndAwait(String chunk, String enc) { 
    return (Void) write(chunk, enc).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#write(String,String)} but you don't need to compose it with other operations.
   * @param chunk 
   * @param enc 
   */
  public void writeAndForget(String chunk, String enc) { 
    write(chunk, enc).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  private io.vertx.mutiny.core.http.HttpClientRequest __continueHandler(Handler<Void> handler) { 
    delegate.continueHandler(handler);
    return this;
  }

  /**
   * @param handler 
   * @return 
   */
  public io.vertx.mutiny.core.http.HttpClientRequest continueHandler(java.lang.Runnable handler) {
    return __continueHandler(ignored -> handler.run()
);
  }

  /**
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  private io.vertx.mutiny.core.http.HttpClientRequest __earlyHintsHandler(Handler<io.vertx.mutiny.core.MultiMap> handler) { 
    delegate.earlyHintsHandler(io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, event -> io.vertx.mutiny.core.MultiMap.newInstance((io.vertx.core.MultiMap)event)));
    return this;
  }

  /**
   * @param handler 
   * @return 
   */
  public io.vertx.mutiny.core.http.HttpClientRequest earlyHintsHandler(java.util.function.Consumer<io.vertx.mutiny.core.MultiMap> handler) {
    return __earlyHintsHandler(io.smallrye.mutiny.vertx.MutinyHelper.convertConsumer(handler));
  }

  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest redirectHandler(Function<io.vertx.mutiny.core.http.HttpClientResponse, io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientRequest>> handler) { 
    delegate.redirectHandler(new java.util.function.Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.HttpClientRequest>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientRequest> apply(io.vertx.core.http.HttpClientResponse arg) {
            return io.smallrye.mutiny.vertx.UniHelper.toFuture(
                 handler.apply(io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)arg)).map(x -> x.getDelegate())
            );
         }
     });
    return this;
  }

  /**
   * Like {@link io.vertx.mutiny.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> sendHead() { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(completionHandler -> {
        delegate.sendHead(completionHandler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#sendHead}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @return the Void instance produced by the operation.
   */
  public Void sendHeadAndAwait() { 
    return (Void) sendHead().await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#sendHead} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#sendHead}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#sendHead} but you don't need to compose it with other operations.
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest sendHeadAndForget() { 
    sendHead().subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.mutiny.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> connect() { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.connect(io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#connect}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse connectAndAwait() { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) connect().await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#connect} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#connect}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#connect} but you don't need to compose it with other operations.
   */
  public void connectAndForget() { 
    connect().subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Set a callback for the associated {@link io.vertx.mutiny.core.http.HttpClientResponse}.
   *
   * <p> This method does not modify the current request being sent.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> response() { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.response(io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#response}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse responseAndAwait() { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) response().await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#response} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#response}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#response} but you don't need to compose it with other operations.
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest responseAndForget() { 
    response().subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
    return this;
  }

  /**
   * Send the request with an empty body.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> send() { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.send(io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse sendAndAwait() { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) send().await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#send}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#send} but you don't need to compose it with other operations.
   */
  public void sendAndForget() { 
    send().subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Send the request with a string <code>body</code>.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param body 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> send(String body) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.send(body, io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param body 
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse sendAndAwait(String body) { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) send(body).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#send(String)} but you don't need to compose it with other operations.
   * @param body 
   */
  public void sendAndForget(String body) { 
    send(body).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param body 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> send(io.vertx.mutiny.core.buffer.Buffer body) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.send(body.getDelegate(), io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.buffer.Buffer)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param body 
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse sendAndAwait(io.vertx.mutiny.core.buffer.Buffer body) { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) send(body).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.buffer.Buffer)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.buffer.Buffer)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.buffer.Buffer)} but you don't need to compose it with other operations.
   * @param body 
   */
  public void sendAndForget(io.vertx.mutiny.core.buffer.Buffer body) { 
    send(body).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.mutiny.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.mutiny.core.http.HttpHeaders}.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param body 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> send(io.vertx.mutiny.core.streams.ReadStream<io.vertx.mutiny.core.buffer.Buffer> body) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.send(body.getDelegate(), io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.streams.ReadStream)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param body 
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse sendAndAwait(io.vertx.mutiny.core.streams.ReadStream<io.vertx.mutiny.core.buffer.Buffer> body) { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) send(body).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.streams.ReadStream)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.streams.ReadStream)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#send(io.vertx.mutiny.core.streams.ReadStream)} but you don't need to compose it with other operations.
   * @param body 
   */
  public void sendAndForget(io.vertx.mutiny.core.streams.ReadStream<io.vertx.mutiny.core.buffer.Buffer> body) { 
    send(body).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.mutiny.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.mutiny.core.http.HttpHeaders}.
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param body 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<io.vertx.mutiny.core.http.HttpClientResponse> send(java.util.concurrent.Flow.Publisher<io.vertx.mutiny.core.buffer.Buffer> body) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.send(io.smallrye.mutiny.vertx.ReadStreamSubscriber.asReadStream(body,obj -> (io.vertx.core.buffer.Buffer) obj.getDelegate()).resume(), io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, ar -> ar.map(event -> io.vertx.mutiny.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(Flow$Publisher)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param body 
   * @return the HttpClientResponse instance produced by the operation.
   */
  public io.vertx.mutiny.core.http.HttpClientResponse sendAndAwait(java.util.concurrent.Flow.Publisher<io.vertx.mutiny.core.buffer.Buffer> body) { 
    return (io.vertx.mutiny.core.http.HttpClientResponse) send(body).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(Flow$Publisher)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#send(Flow$Publisher)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#send(Flow$Publisher)} but you don't need to compose it with other operations.
   * @param body 
   */
  public void sendAndForget(java.util.concurrent.Flow.Publisher<io.vertx.mutiny.core.buffer.Buffer> body) { 
    send(body).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param chunk 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> end(String chunk) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.end(chunk, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param chunk 
   * @return the Void instance produced by the operation.
   */
  public Void endAndAwait(String chunk) { 
    return (Void) end(chunk).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String)} but you don't need to compose it with other operations.
   * @param chunk 
   */
  public void endAndForget(String chunk) { 
    end(chunk).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param chunk 
   * @param enc 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> end(String chunk, String enc) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.end(chunk, enc, handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String,String)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param chunk 
   * @param enc 
   * @return the Void instance produced by the operation.
   */
  public Void endAndAwait(String chunk, String enc) { 
    return (Void) end(chunk, enc).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String,String)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String,String)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#end(String,String)} but you don't need to compose it with other operations.
   * @param chunk 
   * @param enc 
   */
  public void endAndForget(String chunk, String enc) { 
    end(chunk, enc).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @param chunk 
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> end(io.vertx.mutiny.core.buffer.Buffer chunk) { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.end(chunk.getDelegate(), handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(io.vertx.mutiny.core.buffer.Buffer)}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @param chunk 
   * @return the Void instance produced by the operation.
   */
  public Void endAndAwait(io.vertx.mutiny.core.buffer.Buffer chunk) { 
    return (Void) end(chunk).await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(io.vertx.mutiny.core.buffer.Buffer)} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#end(io.vertx.mutiny.core.buffer.Buffer)}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#end(io.vertx.mutiny.core.buffer.Buffer)} but you don't need to compose it with other operations.
   * @param chunk 
   */
  public void endAndForget(io.vertx.mutiny.core.buffer.Buffer chunk) { 
    end(chunk).subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * Same as {@link io.vertx.mutiny.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * <p>
   * Unlike the <em>bare</em> Vert.x variant, this method returns a {@link io.smallrye.mutiny.Uni Uni}.
   * Don't forget to <em>subscribe</em> on it to trigger the operation.
   * @return the {@link io.smallrye.mutiny.Uni uni} firing the result of the operation when completed, or a failure if the operation failed.
   */
  @CheckReturnValue
  public io.smallrye.mutiny.Uni<Void> end() { 
    return io.smallrye.mutiny.vertx.AsyncResultUni.toUni(handler -> {
        delegate.end(handler);
    });
  }

  /**
   * Blocking variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end}.
   * <p>
   * This method waits for the completion of the underlying asynchronous operation.
   * If the operation completes successfully, the result is returned, otherwise the failure is thrown (potentially wrapped in a RuntimeException).
   * @return the Void instance produced by the operation.
   */
  public Void endAndAwait() { 
    return (Void) end().await().indefinitely();
  }

  /**
   * Variant of {@link io.vertx.mutiny.core.http.HttpClientRequest#end} that ignores the result of the operation.
   * <p>
   * This method subscribes on the result of {@link io.vertx.mutiny.core.http.HttpClientRequest#end}, but discards the outcome (item or failure).
   * This method is useful to trigger the asynchronous operation from {@link io.vertx.mutiny.core.http.HttpClientRequest#end} but you don't need to compose it with other operations.
   */
  public void endAndForget() { 
    end().subscribe().with(io.smallrye.mutiny.vertx.UniHelper.NOOP);
  }

  /**
   * @param timeout 
   * @return the instance of HttpClientRequest to chain method calls.
   * @deprecated instead use {@link #idleTimeout(long)}   */
  @Deprecated
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setTimeout(long timeout) { 
    delegate.setTimeout(timeout);
    return this;
  }

  /**
   * @param timeout the amount of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest idleTimeout(long timeout) { 
    delegate.idleTimeout(timeout);
    return this;
  }

  /**
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  private io.vertx.mutiny.core.http.HttpClientRequest __pushHandler(Handler<io.vertx.mutiny.core.http.HttpClientRequest> handler) { 
    delegate.pushHandler(io.smallrye.mutiny.vertx.MutinyHelper.convertHandler(handler, event -> io.vertx.mutiny.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)event)));
    return this;
  }

  /**
   * @param handler the handler
   * @return 
   */
  public io.vertx.mutiny.core.http.HttpClientRequest pushHandler(java.util.function.Consumer<io.vertx.mutiny.core.http.HttpClientRequest> handler) {
    return __pushHandler(io.smallrye.mutiny.vertx.MutinyHelper.convertConsumer(handler));
  }

  /**
   * @return 
   */
  public boolean reset() { 
    boolean ret = delegate.reset();
    return ret;
  }

  /**
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public boolean reset(long code) { 
    boolean ret = delegate.reset(code);
    return ret;
  }

  /**
   * @param code the error code
   * @param cause an optional cause that can be attached to the error code
   * @return true when reset has been performed
   */
  public boolean reset(long code, java.lang.Throwable cause) { 
    boolean ret = delegate.reset(code, cause);
    return ret;
  }

  /**
   * @return the {@link io.vertx.mutiny.core.http.HttpConnection} associated with this request
   */
  public io.vertx.mutiny.core.http.HttpConnection connection() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.mutiny.core.http.HttpConnection ret = io.vertx.mutiny.core.http.HttpConnection.newInstance((io.vertx.core.http.HttpConnection)delegate.connection());
    cached_1 = ret;
    return ret;
  }

  /**
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest writeCustomFrame(int type, int flags, io.vertx.mutiny.core.buffer.Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * @return the id of the stream of this response,  when it is not yet determined, i.e the request has not been yet sent or it is not supported HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * @param frame the frame to write
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest writeCustomFrame(io.vertx.mutiny.core.http.HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  /**
   * @param streamPriority the priority of this request's stream
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest setStreamPriority(io.vertx.core.http.StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * @return the priority of the associated HTTP/2 stream for HTTP/2 otherwise <code>null</code>
   */
  public io.vertx.core.http.StreamPriority getStreamPriority() { 
    io.vertx.core.http.StreamPriority ret = delegate.getStreamPriority();
    return ret;
  }

  /**
   * @param name 
   * @param value 
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @param name The header name
   * @param values The header values
   * @return @return a reference to this, so the API can be used fluently
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest putHeader(String name, java.lang.Iterable<String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * @param name 
   * @param values 
   * @return the instance of HttpClientRequest to chain method calls.
   */
  @Fluent
  public io.vertx.mutiny.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  private io.vertx.mutiny.core.MultiMap cached_0;
  private io.vertx.mutiny.core.http.HttpConnection cached_1;
  public static  HttpClientRequest newInstance(io.vertx.core.http.HttpClientRequest arg) {
    return arg != null ? new HttpClientRequest(arg) : null;
  }

}
