/*
 * Copyright (c)  2016, WSO2 Inc. (http://www.wso2.org) All Rights Reserved.
 *
 * WSO2 Inc. licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package io.siddhi.extension.execution.math;

import io.siddhi.annotation.Example;
import io.siddhi.annotation.Extension;
import io.siddhi.annotation.Parameter;
import io.siddhi.annotation.ReturnAttribute;
import io.siddhi.annotation.util.DataType;
import io.siddhi.core.config.SiddhiQueryContext;
import io.siddhi.core.executor.ExpressionExecutor;
import io.siddhi.core.executor.function.FunctionExecutor;
import io.siddhi.core.util.config.ConfigReader;
import io.siddhi.core.util.snapshot.state.State;
import io.siddhi.core.util.snapshot.state.StateFactory;
import io.siddhi.query.api.definition.Attribute;
import io.siddhi.query.api.exception.SiddhiAppValidationException;

/**
 * oct(a)
 * Converts 'a' to octal
 * Accept Type(s):INT,LONG
 * Return Type(s): STRING
 */
@Extension(
        name = "oct",
        namespace = "math",
        description = "This function converts the input parameter `p1` to octal.",
        parameters = {
                @Parameter(
                        name = "p1",
                        description = "The value of the parameter whose octal representation should be found.",
                        type = {DataType.INT, DataType.LONG})},
        returnAttributes = @ReturnAttribute(
                description = "The octal value of the input parameter.",
                type = {DataType.STRING}),
        examples = @Example(

                syntax = "define stream InValueStream (inValue long); \n" +
                        "from InValueStream \n" +
                        "select math:oct(inValue) as octValue \n" +
                        "insert into OutMediationStream;",
                description = "If the 'inValue' in the input stream is given, this function calculates the " +
                "octal value corresponding to the same and directs it to the output stream, OutMediationStream. " +
                "For example, oct(99l) returns \"143\".")
)
public class OctalFunctionExtension extends FunctionExecutor {

    @Override
    protected StateFactory init(ExpressionExecutor[] expressionExecutors, ConfigReader configReader,
                                SiddhiQueryContext siddhiQueryContext) {
        if (attributeExpressionExecutors.length != 1) {
            throw new SiddhiAppValidationException("Invalid no of arguments passed to math:oct() function, " +
                    "required 1, but found " + attributeExpressionExecutors.length);
        }
        Attribute.Type attributeType = attributeExpressionExecutors[0].getReturnType();
        if (!((attributeType == Attribute.Type.INT)
                || (attributeType == Attribute.Type.LONG))) {
            throw new SiddhiAppValidationException("Invalid parameter type found for the argument of math:oct() " +
                    "function, required " + Attribute.Type.INT + " or " + Attribute.Type.LONG + ", but found " +
                    attributeType.toString());
        }
        return null;
    }

    @Override
    protected Object execute(Object[] data, State state) {
        return null;    // This method won't get called. Hence, unimplemented.
    }

    @Override
    protected Object execute(Object data, State state) {
        if (data != null) {
            if (data instanceof Integer) {
                return Integer.toOctalString((Integer) data);
            } else {
                return Long.toOctalString((Long) data);
            }
        }
        return null;
    }

    @Override
    public Attribute.Type getReturnType() {
        return Attribute.Type.STRING;
    }

}
