/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.core.cache;

import io.sentry.core.ISerializer;
import io.sentry.core.SentryEvent;
import io.sentry.core.SentryLevel;
import io.sentry.core.SentryOptions;
import io.sentry.core.cache.IEventCache;
import io.sentry.core.util.Objects;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

@ApiStatus.Internal
public final class DiskCache
implements IEventCache {
    public static final String FILE_SUFFIX = ".sentry-event";
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    private final File directory;
    private final int maxSize;
    private final ISerializer serializer;
    private final SentryOptions options;

    public DiskCache(SentryOptions options) {
        Objects.requireNonNull(options.getCacheDirPath(), "Cache dir. path is required.");
        this.directory = new File(options.getCacheDirPath());
        this.maxSize = options.getCacheDirSize();
        this.serializer = options.getSerializer();
        this.options = options;
    }

    @Override
    public void store(SentryEvent event) {
        if (this.getNumberOfStoredEvents() >= this.maxSize) {
            this.options.getLogger().log(SentryLevel.WARNING, "Disk cache full (respecting maxSize). Not storing event {}", event);
            return;
        }
        File eventFile = this.getEventFile(event);
        if (eventFile.exists()) {
            this.options.getLogger().log(SentryLevel.WARNING, "Not adding Event to offline storage because it already exists: %s", eventFile.getAbsolutePath());
            return;
        }
        this.options.getLogger().log(SentryLevel.DEBUG, "Adding Event to offline storage: %s", eventFile.getAbsolutePath());
        try (FileOutputStream outputStream = new FileOutputStream(eventFile);
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)outputStream, UTF_8));){
            this.serializer.serialize(event, (Writer)writer);
        }
        catch (Exception e) {
            this.options.getLogger().log(SentryLevel.ERROR, "Error writing Event to offline storage: %s", event.getEventId());
        }
    }

    @Override
    public void discard(SentryEvent event) {
        File eventFile = this.getEventFile(event);
        if (eventFile.exists()) {
            this.options.getLogger().log(SentryLevel.DEBUG, "Discarding event from cache: %s", eventFile.getAbsolutePath());
            if (!eventFile.delete()) {
                this.options.getLogger().log(SentryLevel.ERROR, "Failed to delete Event: %s", eventFile.getAbsolutePath());
            }
        } else {
            this.options.getLogger().log(SentryLevel.DEBUG, "Event was not cached: %s", eventFile.getAbsolutePath());
        }
    }

    private int getNumberOfStoredEvents() {
        return this.allEventFiles().length;
    }

    private boolean isDirectoryValid() {
        if (!(this.directory.isDirectory() && this.directory.canWrite() && this.directory.canRead())) {
            this.options.getLogger().log(SentryLevel.ERROR, "The directory for caching Sentry events is inaccessible.: %s", this.directory.getAbsolutePath());
            return false;
        }
        return true;
    }

    private File getEventFile(SentryEvent event) {
        return new File(this.directory.getAbsolutePath(), event.getEventId().toString() + FILE_SUFFIX);
    }

    @Override
    @NotNull
    public Iterator<SentryEvent> iterator() {
        File[] allCachedEvents = this.allEventFiles();
        ArrayList<SentryEvent> ret = new ArrayList<SentryEvent>(allCachedEvents.length);
        for (File f : allCachedEvents) {
            try (BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(f), UTF_8));){
                ret.add(this.serializer.deserializeEvent(reader));
            }
            catch (FileNotFoundException e) {
                this.options.getLogger().log(SentryLevel.DEBUG, "Event file '%s' disappeared while converting all cached files to events.", f.getAbsolutePath());
            }
            catch (IOException e) {
                this.options.getLogger().log(SentryLevel.ERROR, String.format("Error while reading cached event from file %s", f.getAbsolutePath()), e);
            }
        }
        return ret.iterator();
    }

    private File[] allEventFiles() {
        if (this.isDirectoryValid()) {
            return this.directory.listFiles((__, fileName) -> fileName.endsWith(FILE_SUFFIX));
        }
        return new File[0];
    }
}

