/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.android.core.internal.tombstone;

import androidx.annotation.NonNull;
import io.sentry.SentryEvent;
import io.sentry.SentryLevel;
import io.sentry.SentryStackTraceFactory;
import io.sentry.android.core.internal.tombstone.NativeExceptionMechanism;
import io.sentry.android.core.internal.tombstone.TombstoneProtos;
import io.sentry.android.core.internal.util.NativeEventUtils;
import io.sentry.protocol.DebugImage;
import io.sentry.protocol.DebugMeta;
import io.sentry.protocol.Mechanism;
import io.sentry.protocol.Message;
import io.sentry.protocol.SentryException;
import io.sentry.protocol.SentryStackFrame;
import io.sentry.protocol.SentryStackTrace;
import io.sentry.protocol.SentryThread;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class TombstoneParser
implements Closeable {
    private final InputStream tombstoneStream;
    @NotNull
    private final List<String> inAppIncludes;
    @NotNull
    private final List<String> inAppExcludes;
    @Nullable
    private final String nativeLibraryDir;
    private final Map<String, String> excTypeValueMap = new HashMap<String, String>();

    private static String formatHex(long value) {
        return String.format("0x%x", value);
    }

    public TombstoneParser(@NonNull InputStream tombstoneStream, @NotNull List<String> inAppIncludes, @NotNull List<String> inAppExcludes, @Nullable String nativeLibraryDir) {
        this.tombstoneStream = tombstoneStream;
        this.inAppIncludes = inAppIncludes;
        this.inAppExcludes = inAppExcludes;
        this.nativeLibraryDir = nativeLibraryDir;
        this.excTypeValueMap.put("SIGILL", "IllegalInstruction");
        this.excTypeValueMap.put("SIGTRAP", "Trap");
        this.excTypeValueMap.put("SIGABRT", "Abort");
        this.excTypeValueMap.put("SIGBUS", "BusError");
        this.excTypeValueMap.put("SIGFPE", "FloatingPointException");
        this.excTypeValueMap.put("SIGSEGV", "Segfault");
    }

    @NonNull
    public SentryEvent parse() throws IOException {
        TombstoneProtos.Tombstone tombstone = TombstoneProtos.Tombstone.parseFrom(this.tombstoneStream);
        SentryEvent event = new SentryEvent();
        event.setLevel(SentryLevel.FATAL);
        event.setPlatform("native");
        event.setMessage(this.constructMessage(tombstone));
        event.setDebugMeta(this.createDebugMeta(tombstone));
        event.setExceptions(this.createException(tombstone));
        event.setThreads(this.createThreads(tombstone, (SentryException)Objects.requireNonNull(event.getExceptions()).get(0)));
        return event;
    }

    @NonNull
    private List<SentryThread> createThreads(@NonNull TombstoneProtos.Tombstone tombstone, @NonNull SentryException exc) {
        ArrayList<SentryThread> threads = new ArrayList<SentryThread>();
        for (Map.Entry<Integer, TombstoneProtos.Thread> threadEntry : tombstone.getThreadsMap().entrySet()) {
            TombstoneProtos.Thread threadEntryValue = threadEntry.getValue();
            SentryThread thread = new SentryThread();
            thread.setId(Long.valueOf(threadEntry.getKey().intValue()));
            thread.setName(threadEntryValue.getName());
            SentryStackTrace stacktrace = this.createStackTrace(threadEntryValue);
            thread.setStacktrace(stacktrace);
            if (tombstone.getTid() == threadEntryValue.getId()) {
                thread.setCrashed(Boolean.valueOf(true));
                exc.setStacktrace(stacktrace);
            }
            threads.add(thread);
        }
        return threads;
    }

    @NonNull
    private SentryStackTrace createStackTrace(@NonNull TombstoneProtos.Thread thread) {
        ArrayList<SentryStackFrame> frames = new ArrayList<SentryStackFrame>();
        for (TombstoneProtos.BacktraceFrame frame : thread.getCurrentBacktraceList()) {
            if (frame.getFileName().endsWith("libart.so") || frame.getFileName().startsWith("<anonymous") && frame.getFunctionName().isEmpty()) continue;
            SentryStackFrame stackFrame = new SentryStackFrame();
            stackFrame.setPackage(frame.getFileName());
            stackFrame.setFunction(frame.getFunctionName());
            stackFrame.setInstructionAddr(TombstoneParser.formatHex(frame.getPc()));
            String functionName = frame.getFunctionName();
            @Nullable Boolean inApp = functionName.isEmpty() ? Boolean.FALSE : SentryStackTraceFactory.isInApp((String)functionName, this.inAppIncludes, this.inAppExcludes);
            boolean isInNativeLibraryDir = this.nativeLibraryDir != null && frame.getFileName().startsWith(this.nativeLibraryDir);
            inApp = inApp != null && inApp != false || isInNativeLibraryDir;
            stackFrame.setInApp(inApp);
            frames.add(0, stackFrame);
        }
        SentryStackTrace stacktrace = new SentryStackTrace();
        stacktrace.setFrames(frames);
        stacktrace.setInstructionAddressAdjustment(SentryStackTrace.InstructionAddressAdjustment.NONE);
        HashMap<String, String> registers = new HashMap<String, String>();
        for (TombstoneProtos.Register register : thread.getRegistersList()) {
            registers.put(register.getName(), TombstoneParser.formatHex(register.getU64()));
        }
        stacktrace.setRegisters(registers);
        return stacktrace;
    }

    @NonNull
    private List<SentryException> createException(@NonNull TombstoneProtos.Tombstone tombstone) {
        SentryException exception = new SentryException();
        if (tombstone.hasSignalInfo()) {
            TombstoneProtos.Signal signalInfo = tombstone.getSignalInfo();
            exception.setType(signalInfo.getName());
            exception.setValue(this.excTypeValueMap.get(signalInfo.getName()));
            exception.setMechanism(TombstoneParser.createMechanismFromSignalInfo(signalInfo));
        }
        exception.setThreadId(Long.valueOf(tombstone.getTid()));
        ArrayList<SentryException> exceptions = new ArrayList<SentryException>(1);
        exceptions.add(exception);
        return exceptions;
    }

    @NonNull
    private static Mechanism createMechanismFromSignalInfo(@NonNull TombstoneProtos.Signal signalInfo) {
        Mechanism mechanism = new Mechanism();
        mechanism.setType(NativeExceptionMechanism.TOMBSTONE.getValue());
        mechanism.setHandled(Boolean.valueOf(false));
        mechanism.setSynthetic(Boolean.valueOf(true));
        HashMap<String, Object> meta = new HashMap<String, Object>();
        meta.put("number", signalInfo.getNumber());
        meta.put("name", signalInfo.getName());
        meta.put("code", signalInfo.getCode());
        meta.put("code_name", signalInfo.getCodeName());
        mechanism.setMeta(meta);
        return mechanism;
    }

    @NonNull
    private Message constructMessage(@NonNull TombstoneProtos.Tombstone tombstone) {
        Message message = new Message();
        TombstoneProtos.Signal signalInfo = tombstone.getSignalInfo();
        String command = String.join((CharSequence)" ", tombstone.getCommandLineList());
        if (tombstone.hasSignalInfo()) {
            String abortMessage = tombstone.getAbortMessage();
            message.setFormatted(String.format(Locale.ROOT, "%sFatal signal %s (%d), %s (%d), pid = %d (%s)", !abortMessage.isEmpty() ? abortMessage + ": " : "", signalInfo.getName(), signalInfo.getNumber(), signalInfo.getCodeName(), signalInfo.getCode(), tombstone.getPid(), command));
        } else {
            message.setFormatted(String.format(Locale.ROOT, "Fatal exit pid = %d (%s)", tombstone.getPid(), command));
        }
        return message;
    }

    private DebugMeta createDebugMeta(@NonNull TombstoneProtos.Tombstone tombstone) {
        DebugImage image;
        ArrayList<DebugImage> images = new ArrayList<DebugImage>();
        ModuleAccumulator currentModule = null;
        for (TombstoneProtos.MemoryMapping mapping : tombstone.getMemoryMappingsList()) {
            boolean isFileStart;
            String mappingName;
            if (!mapping.getRead() || (mappingName = mapping.getMappingName()).isEmpty() || mappingName.startsWith("/dev/")) continue;
            boolean hasBuildId = !mapping.getBuildId().isEmpty();
            boolean bl = isFileStart = mapping.getOffset() == 0L;
            if (hasBuildId && isFileStart) {
                DebugImage image2;
                if (currentModule != null && mappingName.equals(currentModule.mappingName)) {
                    currentModule.extendTo(mapping.getEndAddress());
                    continue;
                }
                if (currentModule != null && (image2 = currentModule.toDebugImage()) != null) {
                    images.add(image2);
                }
                currentModule = new ModuleAccumulator(mapping);
                continue;
            }
            if (currentModule == null || !mappingName.equals(currentModule.mappingName)) continue;
            currentModule.extendTo(mapping.getEndAddress());
        }
        if (currentModule != null && (image = currentModule.toDebugImage()) != null) {
            images.add(image);
        }
        DebugMeta debugMeta = new DebugMeta();
        debugMeta.setImages(images);
        return debugMeta;
    }

    @Override
    public void close() throws IOException {
        this.tombstoneStream.close();
    }

    private static class ModuleAccumulator {
        String mappingName;
        String buildId;
        long beginAddress;
        long endAddress;

        ModuleAccumulator(TombstoneProtos.MemoryMapping mapping) {
            this.mappingName = mapping.getMappingName();
            this.buildId = mapping.getBuildId();
            this.beginAddress = mapping.getBeginAddress();
            this.endAddress = mapping.getEndAddress();
        }

        void extendTo(long newEndAddress) {
            this.endAddress = newEndAddress;
        }

        DebugImage toDebugImage() {
            if (this.buildId.isEmpty()) {
                return null;
            }
            DebugImage image = new DebugImage();
            image.setCodeId(this.buildId);
            image.setCodeFile(this.mappingName);
            String debugId = NativeEventUtils.buildIdToDebugId(this.buildId);
            image.setDebugId(debugId != null ? debugId : this.buildId);
            image.setImageAddr(TombstoneParser.formatHex(this.beginAddress));
            image.setImageSize(this.endAddress - this.beginAddress);
            image.setType("elf");
            return image;
        }
    }
}

