/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.tls.cli;

import io.quarkus.tls.cli.Constants;
import io.quarkus.tls.cli.DotEnvHelper;
import io.quarkus.tls.cli.letsencrypt.LetsEncryptConstants;
import io.smallrye.certs.CertificateGenerator;
import io.smallrye.certs.CertificateRequest;
import io.smallrye.certs.Format;
import io.smallrye.common.os.OS;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.Callable;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import picocli.CommandLine;

@CommandLine.Command(name="generate-certificate", mixinStandardHelpOptions=true, description={"Generate a TLS certificate with the Quarkus Dev CA if available."})
public class GenerateCertificateCommand
implements Callable<Integer> {
    @CommandLine.Option(names={"-n", "--name"}, description={"Name of the certificate. It will be used as file name and alias in the keystore"}, required=true)
    String name;
    @CommandLine.Option(names={"-p", "--password"}, description={"The password of the keystore. Default is 'password'"}, defaultValue="password", required=false)
    String password;
    @CommandLine.Option(names={"-c", "--cn"}, description={"The common name of the certificate. Default is 'localhost'"}, defaultValue="localhost", required=false)
    String cn;
    @CommandLine.Option(names={"-d", "--directory"}, description={"The directory in which the certificates will be created. Default is `.certs`"}, defaultValue=".certs")
    Path directory;
    @CommandLine.Option(names={"-r", "--renew"}, description={"Whether existing certificates will need to be replaced"}, defaultValue="false")
    boolean renew;
    @CommandLine.Option(names={"--self-signed"}, description={"Generate a self-signed certificate"}, defaultValue="false", hidden=true)
    boolean selfSigned;
    static System.Logger LOGGER;

    @Override
    public Integer call() throws Exception {
        LOGGER.log(System.Logger.Level.INFO, "\ud83d\udd0e Looking for the Quarkus Dev CA certificate...");
        if (!Constants.CA_FILE.exists() || !Constants.PK_FILE.exists() || this.selfSigned) {
            LOGGER.log(System.Logger.Level.INFO, "\ud83c\udfb2 Quarkus Dev CA certificate not found. Generating a self-signed certificate...");
            this.generateSelfSignedCertificate();
            return 0;
        }
        LOGGER.log(System.Logger.Level.INFO, "\ud83d\udcdc Quarkus Dev CA certificate found at {0}", Constants.CA_FILE.getAbsolutePath());
        X509Certificate caCert = this.loadRootCertificate(Constants.CA_FILE);
        PrivateKey caPrivateKey = this.loadPrivateKey();
        this.createSignedCertificate(caCert, caPrivateKey);
        LOGGER.log(System.Logger.Level.INFO, "\u2705 Signed Certificate generated successfully and exported into `{0}-keystore.p12`", this.name);
        this.printConfig(this.directory.resolve(this.name + "-keystore.p12"), this.password);
        return 0;
    }

    private void generateSelfSignedCertificate() throws Exception {
        if (!Files.exists(this.directory, new LinkOption[0])) {
            Files.createDirectories(this.directory, new FileAttribute[0]);
        }
        new CertificateGenerator(this.directory, this.renew).generate(new CertificateRequest().withName(this.name).withCN(this.cn).withPassword(this.password).withDuration(Duration.ofDays(365L)).withFormat(Format.PKCS12));
        LOGGER.log(System.Logger.Level.INFO, "\u2705 Self-signed certificate generated successfully and exported into `{0}-keystore.p12`", this.name);
        this.printConfig(this.directory.resolve(this.name + "-keystore.p12"), this.password);
    }

    private void printConfig(Path certificatePath, String password) {
        String certificatePathProperty = certificatePath.toString();
        if (OS.WINDOWS.isCurrent()) {
            certificatePathProperty = certificatePathProperty.replace("\\", "\\\\");
        }
        try {
            List<String> dotEnvContent = DotEnvHelper.readDotEnvFile();
            DotEnvHelper.addOrReplaceProperty(dotEnvContent, "%dev.quarkus.tls.key-store.p12.path", certificatePathProperty);
            DotEnvHelper.addOrReplaceProperty(dotEnvContent, "%dev.quarkus.tls.key-store.p12.password", password);
            Files.write(LetsEncryptConstants.DOT_ENV_FILE.toPath(), dotEnvContent, new OpenOption[0]);
        }
        catch (IOException e) {
            LOGGER.log(System.Logger.Level.ERROR, "Failed to read .env file", (Throwable)e);
        }
        LOGGER.log(System.Logger.Level.INFO, "\u2705 Required configuration added to the `.env` file:\n%dev.quarkus.tls.key-store.p12.path={0}\n%dev.quarkus.tls.key-store.p12.password={1}\n", certificatePathProperty, password);
    }

    private X509Certificate loadRootCertificate(File ca) throws Exception {
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        try (FileInputStream fis = new FileInputStream(ca);){
            X509Certificate x509Certificate = (X509Certificate)cf.generateCertificate(fis);
            return x509Certificate;
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private PrivateKey loadPrivateKey() throws Exception {
        try (BufferedReader reader = new BufferedReader(new FileReader(Constants.PK_FILE));){
            PEMParser pemParser = new PEMParser(reader);
            Object obj = pemParser.readObject();
            if (obj instanceof KeyPair) {
                PrivateKey privateKey = ((KeyPair)obj).getPrivate();
                return privateKey;
            }
            if (!(obj instanceof PrivateKeyInfo)) throw new IllegalStateException("The file " + Constants.PK_FILE.getAbsolutePath() + " does not contain a private key " + obj.getClass().getName());
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter();
            PrivateKey privateKey = converter.getPrivateKey((PrivateKeyInfo)obj);
            return privateKey;
            finally {
                pemParser.close();
            }
        }
    }

    private void createSignedCertificate(X509Certificate issuerCert, PrivateKey issuerPrivateKey) throws Exception {
        if (!Files.exists(this.directory, new LinkOption[0])) {
            Files.createDirectories(this.directory, new FileAttribute[0]);
        }
        new CertificateGenerator(this.directory, this.renew).generate(new CertificateRequest().withName(this.name).withCN(this.cn).withPassword(this.password).withDuration(Duration.ofDays(365L)).withFormat(Format.PKCS12).signedWith(issuerCert, issuerPrivateKey));
    }

    static {
        Security.addProvider(new BouncyCastleProvider());
        LOGGER = System.getLogger("generate-certificate");
    }
}

