/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.deployment.util;

import io.smallrye.config.SmallRyeConfig;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.SystemUtils;
import org.eclipse.microprofile.config.ConfigProvider;
import org.jboss.logging.Logger;

public final class ContainerRuntimeUtil {
    private static final Logger log = Logger.getLogger(ContainerRuntimeUtil.class);
    private static final String CONTAINER_EXECUTABLE = ((SmallRyeConfig)ConfigProvider.getConfig().unwrap(SmallRyeConfig.class)).getOptionalValue("quarkus.native.container-runtime", String.class).orElse(null);
    private static final Pattern PODMAN_PATTERN = Pattern.compile("^podman(?:\\.exe)? version.*", 32);
    private static final String CONTAINER_RUNTIME_SYS_PROP = "quarkus-local-container-runtime";
    private static final int MAX_ANTICIPATED_CHARACTERS_IN_DOCKER_INFO = 3000;

    private ContainerRuntimeUtil() {
    }

    public static ContainerRuntime detectContainerRuntime() {
        return ContainerRuntimeUtil.detectContainerRuntime(true);
    }

    public static ContainerRuntime detectContainerRuntime(boolean required) {
        ContainerRuntime containerRuntime = ContainerRuntimeUtil.loadContainerRuntimeFromSystemProperty();
        if (containerRuntime != null) {
            return containerRuntime;
        }
        ContainerRuntime containerRuntimeEnvironment = ContainerRuntimeUtil.getContainerRuntimeEnvironment();
        if (containerRuntimeEnvironment == ContainerRuntime.UNAVAILABLE) {
            ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(ContainerRuntime.UNAVAILABLE);
            if (required) {
                throw new IllegalStateException("No container runtime was found. Make sure you have either Docker or Podman installed in your environment.");
            }
            return ContainerRuntime.UNAVAILABLE;
        }
        containerRuntime = ContainerRuntimeUtil.fullyResolveContainerRuntime(containerRuntimeEnvironment);
        ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(containerRuntime);
        return containerRuntime;
    }

    private static ContainerRuntime getContainerRuntimeEnvironment() {
        String podmanVersionOutput;
        boolean podmanAvailable;
        String dockerVersionOutput;
        boolean dockerAvailable;
        if (CONTAINER_EXECUTABLE != null) {
            if (CONTAINER_EXECUTABLE.trim().equalsIgnoreCase("docker") && (dockerAvailable = (dockerVersionOutput = ContainerRuntimeUtil.getVersionOutputFor(ContainerRuntime.DOCKER)).contains("Docker version"))) {
                return ContainerRuntime.DOCKER;
            }
            if (CONTAINER_EXECUTABLE.trim().equalsIgnoreCase("podman") && (podmanAvailable = PODMAN_PATTERN.matcher(podmanVersionOutput = ContainerRuntimeUtil.getVersionOutputFor(ContainerRuntime.PODMAN)).matches())) {
                return ContainerRuntime.PODMAN;
            }
            log.warn((Object)"quarkus.native.container-runtime config property must be set to either podman or docker and the executable must be available. Ignoring it.");
        }
        if (dockerAvailable = (dockerVersionOutput = ContainerRuntimeUtil.getVersionOutputFor(ContainerRuntime.DOCKER)).contains("Docker version")) {
            if (PODMAN_PATTERN.matcher(dockerVersionOutput).matches()) {
                return ContainerRuntime.PODMAN;
            }
            return ContainerRuntime.DOCKER;
        }
        podmanVersionOutput = ContainerRuntimeUtil.getVersionOutputFor(ContainerRuntime.PODMAN);
        podmanAvailable = PODMAN_PATTERN.matcher(podmanVersionOutput).matches();
        if (podmanAvailable) {
            return ContainerRuntime.PODMAN;
        }
        return ContainerRuntime.UNAVAILABLE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ContainerRuntime fullyResolveContainerRuntime(ContainerRuntime containerRuntimeEnvironment) {
        boolean rootless = false;
        boolean isInWindowsWSL = false;
        Process rootlessProcess = null;
        ProcessBuilder pb = null;
        try {
            pb = new ProcessBuilder(containerRuntimeEnvironment.getExecutableName(), "info").redirectErrorStream(true);
            rootlessProcess = pb.start();
            int exitCode = rootlessProcess.waitFor();
            if (exitCode != 0) {
                log.warnf("Command \"%s\" exited with error code %d. Rootless container runtime detection might not be reliable or the container service is not running at all.", (Object)String.join((CharSequence)" ", pb.command()), (Object)exitCode);
            }
            try (InputStream inputStream = rootlessProcess.getInputStream();
                 InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
                 BufferedReader bufferedReader = new BufferedReader(inputStreamReader);){
                bufferedReader.mark(3000);
                if (exitCode != 0) {
                    log.debugf("Command \"%s\" output: %s", (Object)String.join((CharSequence)" ", pb.command()), (Object)bufferedReader.lines().collect(Collectors.joining(System.lineSeparator())));
                } else {
                    Predicate<String> stringPredicate = containerRuntimeEnvironment.isDocker() ? line -> line.trim().equals("rootless") || line.contains("Docker Desktop") || line.contains("desktop-linux") : line -> line.trim().equals("rootless: true");
                    rootless = bufferedReader.lines().anyMatch(stringPredicate);
                    if (SystemUtils.IS_OS_LINUX && containerRuntimeEnvironment.isDocker()) {
                        stringPredicate = line -> line.trim().contains("WSL");
                        bufferedReader.reset();
                        isInWindowsWSL = bufferedReader.lines().anyMatch(stringPredicate);
                    }
                }
            }
        }
        catch (IOException | InterruptedException e) {
            log.debugf((Throwable)e, "Failure to read info output from %s", (Object)String.join((CharSequence)" ", pb.command()));
        }
        finally {
            if (rootlessProcess != null) {
                rootlessProcess.destroy();
            }
        }
        if (rootless) {
            if (isInWindowsWSL) {
                return ContainerRuntime.WSL_ROOTLESS;
            }
            return containerRuntimeEnvironment == ContainerRuntime.DOCKER ? ContainerRuntime.DOCKER_ROOTLESS : ContainerRuntime.PODMAN_ROOTLESS;
        }
        if (isInWindowsWSL) {
            return ContainerRuntime.WSL;
        }
        return containerRuntimeEnvironment;
    }

    private static ContainerRuntime loadContainerRuntimeFromSystemProperty() {
        String runtime = System.getProperty(CONTAINER_RUNTIME_SYS_PROP);
        if (runtime == null) {
            return null;
        }
        ContainerRuntime containerRuntime = ContainerRuntime.of(runtime);
        if (containerRuntime == null) {
            log.warnf("System property %s contains an unknown value %s. Ignoring it.", (Object)CONTAINER_RUNTIME_SYS_PROP, (Object)runtime);
        }
        return containerRuntime;
    }

    private static void storeContainerRuntimeInSystemProperty(ContainerRuntime containerRuntime) {
        System.setProperty(CONTAINER_RUNTIME_SYS_PROP, containerRuntime.name());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String getVersionOutputFor(ContainerRuntime containerRuntime) {
        Process versionProcess = null;
        try {
            ProcessBuilder pb = new ProcessBuilder(containerRuntime.getExecutableName(), "--version").redirectErrorStream(true);
            versionProcess = pb.start();
            int timeoutS = 10;
            if (versionProcess.waitFor(10L, TimeUnit.SECONDS)) {
                String string = new String(versionProcess.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
                return string;
            }
            log.debugf("Failure. It took command %s more than %d seconds to execute.", (Object)containerRuntime.getExecutableName(), (Object)10);
            String string = "";
            return string;
        }
        catch (IOException | InterruptedException e) {
            log.debugf((Throwable)e, "Failure to read version output from %s", (Object)containerRuntime.getExecutableName());
            String string = "";
            return string;
        }
        finally {
            if (versionProcess != null) {
                versionProcess.destroy();
            }
        }
    }

    public static enum ContainerRuntime {
        DOCKER("docker", false),
        DOCKER_ROOTLESS("docker", true),
        WSL("docker", false),
        WSL_ROOTLESS("docker", false),
        PODMAN("podman", false),
        PODMAN_ROOTLESS("podman", true),
        UNAVAILABLE(null, false);

        private final String executableName;
        private final boolean rootless;

        private ContainerRuntime(String executableName, boolean rootless) {
            this.executableName = executableName;
            this.rootless = rootless;
        }

        public String getExecutableName() {
            if (this == UNAVAILABLE) {
                throw new IllegalStateException("Cannot get an executable name when no container runtime is available");
            }
            return this.executableName;
        }

        public boolean isDocker() {
            return this.executableName.equals("docker");
        }

        public boolean isPodman() {
            return this == PODMAN || this == PODMAN_ROOTLESS;
        }

        public boolean isInWindowsWSL() {
            return this == WSL || this == WSL_ROOTLESS;
        }

        public boolean isRootless() {
            return this.rootless;
        }

        public static ContainerRuntime of(String value) {
            for (ContainerRuntime containerRuntime : ContainerRuntime.values()) {
                if (!containerRuntime.name().equalsIgnoreCase(value)) continue;
                return containerRuntime;
            }
            return null;
        }
    }
}

