/*
 * Copyright 2019, OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.opentelemetry.sdk.trace.export;

import io.opentelemetry.sdk.trace.TracerSdkProvider;
import io.opentelemetry.sdk.trace.data.SpanData;
import java.util.Collection;

/**
 * An interface that allows different tracing services to export recorded data for sampled spans in
 * their own format.
 *
 * <p>To export data this MUST be register to the {@code TracerSdk} using a {@link
 * SimpleSpansProcessor} or a {@code BatchSampledSpansProcessor}.
 */
public interface SpanExporter {

  /** The possible results for the export method. */
  enum ResultCode {
    /** The export operation finished successfully. */
    SUCCESS,

    /** The export operation finished with an error. */
    FAILURE
  }

  /**
   * Called to export sampled {@code Span}s.
   *
   * @param spans the collection of sampled Spans to be exported.
   * @return the result of the export.
   */
  ResultCode export(Collection<SpanData> spans);

  /**
   * Exports the collection of sampled {@code Span}s that have not yet been exported.
   *
   * @return the result of the flush.
   * @since 0.4.0
   */
  ResultCode flush();

  /**
   * Called when {@link TracerSdkProvider#shutdown()} is called, if this {@code SpanExporter} is
   * register to a {@code TracerSdkProvider} object.
   */
  void shutdown();
}
