/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.exporter.otlp.internal.okhttp;

import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.exporter.otlp.internal.ExporterMetrics;
import io.opentelemetry.exporter.otlp.internal.Marshaler;
import io.opentelemetry.exporter.otlp.internal.grpc.GrpcStatusUtil;
import io.opentelemetry.exporter.otlp.internal.okhttp.ProtoRequestBody;
import io.opentelemetry.exporter.otlp.internal.retry.RetryUtil;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.ThrottlingLogger;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Headers;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okio.BufferedSink;
import okio.GzipSink;
import okio.Okio;
import okio.Sink;

public final class OkHttpExporter<T extends Marshaler> {
    private static final Logger internalLogger = Logger.getLogger(OkHttpExporter.class.getName());
    private final ThrottlingLogger logger = new ThrottlingLogger(internalLogger);
    private final String type;
    private final OkHttpClient client;
    private final String endpoint;
    @Nullable
    private final Headers headers;
    private final boolean compressionEnabled;
    private final ExporterMetrics exporterMetrics;

    OkHttpExporter(String type, OkHttpClient client, MeterProvider meterProvider, String endpoint, @Nullable Headers headers, boolean compressionEnabled) {
        this.type = type;
        this.client = client;
        this.endpoint = endpoint;
        this.headers = headers;
        this.compressionEnabled = compressionEnabled;
        this.exporterMetrics = ExporterMetrics.createHttpProtobuf(type, meterProvider);
    }

    public CompletableResultCode export(T exportRequest, final int numItems) {
        this.exporterMetrics.addSeen(numItems);
        Request.Builder requestBuilder = new Request.Builder().url(this.endpoint);
        if (this.headers != null) {
            requestBuilder.headers(this.headers);
        }
        ProtoRequestBody requestBody = new ProtoRequestBody((Marshaler)exportRequest);
        if (this.compressionEnabled) {
            requestBuilder.addHeader("Content-Encoding", "gzip");
            requestBuilder.post(OkHttpExporter.gzipRequestBody(requestBody));
        } else {
            requestBuilder.post((RequestBody)requestBody);
        }
        final CompletableResultCode result = new CompletableResultCode();
        this.client.newCall(requestBuilder.build()).enqueue(new Callback(){

            public void onFailure(Call call, IOException e) {
                OkHttpExporter.this.exporterMetrics.addFailed(numItems);
                OkHttpExporter.this.logger.log(Level.SEVERE, "Failed to export " + OkHttpExporter.this.type + "s. The request could not be executed. Full error message: " + e.getMessage());
                result.fail();
            }

            public void onResponse(Call call, Response response) {
                try (ResponseBody body = response.body();){
                    if (response.isSuccessful()) {
                        OkHttpExporter.this.exporterMetrics.addSuccess(numItems);
                        result.succeed();
                        return;
                    }
                    OkHttpExporter.this.exporterMetrics.addFailed(numItems);
                    int code = response.code();
                    String status = OkHttpExporter.extractErrorStatus(response, body);
                    OkHttpExporter.this.logger.log(Level.WARNING, "Failed to export " + OkHttpExporter.this.type + "s. Server responded with HTTP status code " + code + ". Error message: " + status);
                    result.fail();
                }
            }
        });
        return result;
    }

    public CompletableResultCode shutdown() {
        CompletableResultCode result = CompletableResultCode.ofSuccess();
        this.client.dispatcher().cancelAll();
        this.client.dispatcher().executorService().shutdownNow();
        this.client.connectionPool().evictAll();
        return result;
    }

    static boolean isRetryable(Response response) {
        return RetryUtil.retryableHttpResponseCodes().contains(response.code());
    }

    private static RequestBody gzipRequestBody(final RequestBody requestBody) {
        return new RequestBody(){

            public MediaType contentType() {
                return requestBody.contentType();
            }

            public long contentLength() {
                return -1L;
            }

            public void writeTo(BufferedSink bufferedSink) throws IOException {
                BufferedSink gzipSink = Okio.buffer((Sink)new GzipSink((Sink)bufferedSink));
                requestBody.writeTo(gzipSink);
                gzipSink.close();
            }
        };
    }

    private static String extractErrorStatus(Response response, @Nullable ResponseBody responseBody) {
        if (responseBody == null) {
            return "Response body missing, HTTP status message: " + response.message();
        }
        try {
            return GrpcStatusUtil.getStatusMessage(responseBody.bytes());
        }
        catch (IOException e) {
            return "Unable to parse response body, HTTP status message: " + response.message();
        }
    }
}

