/*
 * Decompiled with CFR 0.152.
 */
package io.opentelemetry.contrib.inferredspans.internal.asyncprofiler;

import io.opentelemetry.contrib.inferredspans.WildcardMatcher;
import io.opentelemetry.contrib.inferredspans.internal.StackFrame;
import io.opentelemetry.contrib.inferredspans.internal.asyncprofiler.BufferedFile;
import io.opentelemetry.contrib.inferredspans.internal.pooling.Recyclable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import org.agrona.collections.Int2IntHashMap;
import org.agrona.collections.Int2ObjectHashMap;
import org.agrona.collections.Long2LongHashMap;
import org.agrona.collections.Long2ObjectHashMap;

public class JfrParser
implements Recyclable {
    private static final Logger logger = Logger.getLogger(JfrParser.class.getName());
    private static final byte[] MAGIC_BYTES = new byte[]{70, 76, 82, 0};
    private static final Set<String> JAVA_FRAME_TYPES = new HashSet<String>(Arrays.asList("Interpreted", "JIT compiled", "Inlined"));
    private static final int BIG_FILE_BUFFER_SIZE = 0x500000;
    private static final int SMALL_FILE_BUFFER_SIZE = 4096;
    private static final String SYMBOL_EXCLUDED = "3x cluded";
    private static final String SYMBOL_NULL = "n u11";
    private static final StackFrame FRAME_EXCLUDED = new StackFrame("excluded", "excluded");
    private static final StackFrame FRAME_NULL = new StackFrame("null", "null");
    private final BufferedFile bufferedFile;
    private final Int2IntHashMap classIdToClassNameSymbolId = new Int2IntHashMap(-1);
    private final Int2IntHashMap symbolIdToPos = new Int2IntHashMap(-1);
    private final Int2ObjectHashMap<String> symbolIdToString = new Int2ObjectHashMap();
    private final Int2IntHashMap stackTraceIdToFilePositions = new Int2IntHashMap(-1);
    private final Long2LongHashMap nativeTidToJavaTid = new Long2LongHashMap(-1L);
    private final Long2ObjectHashMap<StackFrame> methodIdToFrame = new Long2ObjectHashMap();
    private final Long2LongHashMap methodIdToMethodNameSymbol = new Long2LongHashMap(-1L);
    private final Long2LongHashMap methodIdToClassId = new Long2LongHashMap(-1L);
    private final StringBuilder symbolBuilder = new StringBuilder();
    private long eventsFilePosition;
    private long metadataFilePosition;
    @Nullable
    private boolean[] isJavaFrameType;
    @Nullable
    private List<WildcardMatcher> excludedClasses;
    @Nullable
    private List<WildcardMatcher> includedClasses;

    public JfrParser() {
        this(ByteBuffer.allocateDirect(0x500000), ByteBuffer.allocateDirect(4096));
    }

    JfrParser(ByteBuffer bigBuffer, ByteBuffer smallBuffer) {
        this.bufferedFile = new BufferedFile(bigBuffer, smallBuffer);
    }

    public void parse(File file, List<WildcardMatcher> excludedClasses, List<WildcardMatcher> includedClasses) throws IOException {
        this.excludedClasses = excludedClasses;
        this.includedClasses = includedClasses;
        this.bufferedFile.setFile(file);
        long fileSize = this.bufferedFile.size();
        int chunkSize = this.readChunk(0);
        if ((long)chunkSize < fileSize) {
            throw new IllegalStateException("This implementation does not support reading JFR files containing multiple chunks");
        }
    }

    private int readChunk(int position) throws IOException {
        this.bufferedFile.position(position);
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Parsing JFR chunk at offset", new Object[]{position});
        }
        for (byte magicByte : MAGIC_BYTES) {
            if (this.bufferedFile.get() == magicByte) continue;
            throw new IllegalArgumentException("Not a JFR file");
        }
        short major = this.bufferedFile.getShort();
        short minor = this.bufferedFile.getShort();
        if (major != 2 || minor != 0) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Can only parse version 2.0. Was %d.%d", major, minor));
        }
        long chunkSize = this.bufferedFile.getLong();
        long constantPoolOffset = this.bufferedFile.getLong();
        this.metadataFilePosition = (long)position + this.bufferedFile.getLong();
        this.bufferedFile.getLong();
        this.bufferedFile.getLong();
        this.bufferedFile.getLong();
        this.bufferedFile.getLong();
        this.bufferedFile.getInt();
        this.eventsFilePosition = this.metadataFilePosition + this.parseMetadata(this.metadataFilePosition);
        this.parseCheckpointEvents((long)position + constantPoolOffset);
        return (int)chunkSize;
    }

    private long parseMetadata(long metadataOffset) throws IOException {
        this.bufferedFile.position(metadataOffset);
        int size = this.bufferedFile.getVarInt();
        this.expectEventType(0);
        return size;
    }

    private void expectEventType(int expectedEventType) throws IOException {
        long eventType = this.bufferedFile.getVarLong();
        if (eventType != (long)expectedEventType) {
            throw new IOException("Expected " + expectedEventType + " but got " + eventType);
        }
    }

    private void parseCheckpointEvents(long checkpointOffset) throws IOException {
        this.bufferedFile.position(checkpointOffset);
        this.bufferedFile.getVarInt();
        this.expectEventType(1);
        this.bufferedFile.getVarLong();
        this.bufferedFile.getVarLong();
        long delta = this.bufferedFile.getVarLong();
        if (delta != 0L) {
            throw new IllegalStateException("Expected only one checkpoint event, but file contained multiple, delta is " + delta);
        }
        this.bufferedFile.get();
        long poolCount = this.bufferedFile.getVarLong();
        int i = 0;
        while ((long)i < poolCount) {
            this.parseConstantPool();
            ++i;
        }
    }

    private void parseConstantPool() throws IOException {
        long typeId = this.bufferedFile.getVarLong();
        int count = this.bufferedFile.getVarInt();
        switch ((int)typeId) {
            case 24: {
                this.readFrameTypeConstants(count);
                break;
            }
            case 25: 
            case 32: 
            case 33: {
                for (int i = 0; i < count; ++i) {
                    this.bufferedFile.getVarInt();
                    this.bufferedFile.skipString();
                }
                break;
            }
            case 22: {
                this.readThreadConstants(count);
                break;
            }
            case 26: {
                this.readStackTraceConstants(count);
                break;
            }
            case 28: {
                this.readMethodConstants(count);
                break;
            }
            case 21: {
                this.readClassConstants(count);
                break;
            }
            case 30: {
                this.readPackageConstants(count);
                break;
            }
            case 31: {
                this.readSymbolConstants(count);
                break;
            }
            case 20: {
                this.bufferedFile.skip(2);
                break;
            }
            default: {
                throw new IllegalStateException("Unhandled constant pool type: " + typeId);
            }
        }
    }

    private void readSymbolConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            int symbolId = this.bufferedFile.getVarInt();
            int pos = (int)this.bufferedFile.position();
            this.bufferedFile.skipString();
            this.symbolIdToPos.put(symbolId, pos);
            this.symbolIdToString.put(symbolId, (Object)SYMBOL_NULL);
        }
    }

    private void readClassConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            int classId = this.bufferedFile.getVarInt();
            this.bufferedFile.getVarInt();
            int classNameSymbolId = this.bufferedFile.getVarInt();
            this.classIdToClassNameSymbolId.put(classId, classNameSymbolId);
            this.bufferedFile.getVarInt();
            this.bufferedFile.getVarInt();
        }
    }

    private void readMethodConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            long id = this.bufferedFile.getVarLong();
            int classId = this.bufferedFile.getVarInt();
            int methodNameSymbolId = this.bufferedFile.getVarInt();
            this.methodIdToFrame.put(id, (Object)FRAME_NULL);
            this.methodIdToClassId.put(id, (long)classId);
            this.methodIdToMethodNameSymbol.put(id, (long)methodNameSymbolId);
            this.bufferedFile.getVarLong();
            this.bufferedFile.getVarInt();
            this.bufferedFile.get();
        }
    }

    private void readPackageConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            this.bufferedFile.getVarLong();
            this.bufferedFile.getVarLong();
        }
    }

    private void readThreadConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            int nativeThreadId = this.bufferedFile.getVarInt();
            this.bufferedFile.skipString();
            this.bufferedFile.getVarInt();
            this.bufferedFile.skipString();
            long javaThreadId = this.bufferedFile.getVarLong();
            if (javaThreadId == 0L) continue;
            this.nativeTidToJavaTid.put((long)nativeThreadId, javaThreadId);
        }
    }

    private void readStackTraceConstants(int count) throws IOException {
        for (int i = 0; i < count; ++i) {
            int stackTraceId = this.bufferedFile.getVarInt();
            this.bufferedFile.get();
            this.stackTraceIdToFilePositions.put(stackTraceId, (int)this.bufferedFile.position());
            this.readOrSkipStacktraceFrames(null, 0);
        }
    }

    private void readFrameTypeConstants(int count) throws IOException {
        this.isJavaFrameType = new boolean[count];
        for (int i = 0; i < count; ++i) {
            int id = this.bufferedFile.getVarInt();
            if (i != id) {
                throw new IllegalStateException("Expecting ids to be incrementing");
            }
            this.isJavaFrameType[id] = JAVA_FRAME_TYPES.contains(this.bufferedFile.readString());
        }
    }

    public void consumeStackTraces(StackTraceConsumer callback) throws IOException {
        int eventSize;
        if (!this.bufferedFile.isSet()) {
            throw new IllegalStateException("consumeStackTraces was called before parse");
        }
        this.bufferedFile.position(this.eventsFilePosition);
        long fileSize = this.bufferedFile.size();
        for (long eventStart = this.eventsFilePosition; eventStart < fileSize; eventStart += (long)eventSize) {
            this.bufferedFile.position(eventStart);
            eventSize = this.bufferedFile.getVarInt();
            long eventType = this.bufferedFile.getVarLong();
            if (eventType != 101L) continue;
            long nanoTime = this.bufferedFile.getVarLong();
            int tid = this.bufferedFile.getVarInt();
            int stackTraceId = this.bufferedFile.getVarInt();
            this.bufferedFile.getVarInt();
            long javaThreadId = this.nativeTidToJavaTid.get((long)tid);
            callback.onCallTree(javaThreadId, stackTraceId, nanoTime);
        }
    }

    public void resolveStackTrace(long stackTraceId, List<StackFrame> stackFrames, int maxStackDepth) throws IOException {
        if (!this.bufferedFile.isSet()) {
            throw new IllegalStateException("getStackTrace was called before parse");
        }
        this.bufferedFile.position(this.stackTraceIdToFilePositions.get((int)stackTraceId));
        this.readOrSkipStacktraceFrames(stackFrames, maxStackDepth);
    }

    private void readOrSkipStacktraceFrames(@Nullable List<StackFrame> stackFrames, int maxStackDepth) throws IOException {
        int frameCount = this.bufferedFile.getVarInt();
        for (int i = 0; i < frameCount; ++i) {
            int methodId = this.bufferedFile.getVarInt();
            this.bufferedFile.getVarInt();
            this.bufferedFile.getVarInt();
            byte type = this.bufferedFile.get();
            if (stackFrames == null) continue;
            this.addFrameIfIncluded(stackFrames, methodId, type);
            if (stackFrames.size() <= maxStackDepth) continue;
            stackFrames.remove(0);
        }
    }

    private void addFrameIfIncluded(List<StackFrame> stackFrames, int methodId, byte frameType) throws IOException {
        StackFrame stackFrame;
        if (this.isJavaFrameType(frameType) && (stackFrame = this.resolveStackFrame(methodId)) != FRAME_EXCLUDED) {
            stackFrames.add(stackFrame);
        }
    }

    private boolean isJavaFrameType(byte frameType) {
        assert (this.isJavaFrameType != null);
        return this.isJavaFrameType[frameType];
    }

    private String resolveSymbol(int id, boolean classSymbol) throws IOException {
        String symbol = (String)this.symbolIdToString.get(id);
        if (symbol != SYMBOL_NULL) {
            return symbol;
        }
        long previousPosition = this.bufferedFile.position();
        int position = this.symbolIdToPos.get(id);
        this.bufferedFile.position(position);
        this.symbolBuilder.setLength(0);
        this.bufferedFile.readString(this.symbolBuilder);
        this.bufferedFile.position(previousPosition);
        if (classSymbol) {
            JfrParser.replaceSlashesWithDots(this.symbolBuilder);
        }
        symbol = classSymbol && !this.isClassIncluded(this.symbolBuilder) ? SYMBOL_EXCLUDED : this.symbolBuilder.toString();
        this.symbolIdToString.put(id, (Object)symbol);
        return symbol;
    }

    private static void replaceSlashesWithDots(StringBuilder builder) {
        for (int i = 0; i < builder.length(); ++i) {
            if (builder.charAt(i) != '/') continue;
            builder.setCharAt(i, '.');
        }
    }

    private boolean isClassIncluded(CharSequence className) {
        assert (this.includedClasses != null);
        assert (this.excludedClasses != null);
        return WildcardMatcher.isAnyMatch(this.includedClasses, className) && WildcardMatcher.isNoneMatch(this.excludedClasses, className);
    }

    private StackFrame resolveStackFrame(long frameId) throws IOException {
        StackFrame stackFrame = (StackFrame)this.methodIdToFrame.get(frameId);
        if (stackFrame != FRAME_NULL) {
            return stackFrame;
        }
        String className = this.resolveSymbol(this.classIdToClassNameSymbolId.get((int)this.methodIdToClassId.get(frameId)), true);
        if (className == SYMBOL_EXCLUDED) {
            stackFrame = FRAME_EXCLUDED;
        } else {
            String method = this.resolveSymbol((int)this.methodIdToMethodNameSymbol.get(frameId), false);
            stackFrame = new StackFrame(className, Objects.requireNonNull(method));
        }
        this.methodIdToFrame.put(frameId, (Object)stackFrame);
        return stackFrame;
    }

    @Override
    public void resetState() {
        this.bufferedFile.resetState();
        this.eventsFilePosition = 0L;
        this.metadataFilePosition = 0L;
        this.isJavaFrameType = null;
        this.classIdToClassNameSymbolId.clear();
        this.stackTraceIdToFilePositions.clear();
        this.methodIdToFrame.clear();
        this.methodIdToMethodNameSymbol.clear();
        this.methodIdToClassId.clear();
        this.symbolBuilder.setLength(0);
        this.excludedClasses = null;
        this.includedClasses = null;
        this.symbolIdToPos.clear();
        this.symbolIdToString.clear();
    }

    private static class EventTypeId {
        static final int EVENT_METADATA = 0;
        static final int EVENT_CHECKPOINT = 1;
        static final int EVENT_EXECUTION_SAMPLE = 101;

        private EventTypeId() {
        }
    }

    public static interface StackTraceConsumer {
        public void onCallTree(long var1, long var3, long var5) throws IOException;
    }

    private static final class ContentTypeId {
        static final int CONTENT_THREAD = 22;
        static final int CONTENT_LOG_LEVELS = 33;
        static final int CONTENT_STACKTRACE = 26;
        static final int CONTENT_CLASS = 21;
        static final int CONTENT_METHOD = 28;
        static final int CONTENT_SYMBOL = 31;
        static final int CONTENT_THREAD_STATE = 25;
        static final int CONTENT_FRAME_TYPE = 24;
        static final int CONTENT_GC_WHEN = 32;
        static final int CONTENT_PACKAGE = 30;
        static final int CONTENT_STRING = 20;

        private ContentTypeId() {
        }
    }
}

