/*
 * Copyright 2003-2012 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.gradle.docker.tasks;

import org.gradle.api.DefaultTask;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.internal.file.FileOperations;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.OutputDirectory;
import org.gradle.api.tasks.TaskAction;

import javax.inject.Inject;
import java.io.File;

/**
 * Prepares a docker build context directory by copying a number of
 * directories into the output directory.
 *
 * We cannot use the regular Copy task because of laziness and the
 * fact that the directories to copy are generated by the build itself.
 *
 * However, using `Copy` task is considered harmful in general.
 */
public abstract class PrepareDockerContext extends DefaultTask {
    @Inject
    protected abstract FileOperations getFileOperations();

    @InputFiles
    public abstract ConfigurableFileCollection getInputDirectories();

    @OutputDirectory
    public abstract DirectoryProperty getOutputDirectory();

    @TaskAction
    void copy() {
        for (File directory : getInputDirectories().getFiles()) {
            getFileOperations().copy(spec -> spec.into(getOutputDirectory().dir(directory.getName())).from(directory));
        }
    }
}
