/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.bitbucket.jenkins.internal.applink.oauth.serviceprovider.consumer;

import com.atlassian.bitbucket.jenkins.internal.applink.oauth.serviceprovider.AbstractPersistentStore;
import com.atlassian.bitbucket.jenkins.internal.applink.oauth.serviceprovider.consumer.Consumer;
import com.atlassian.bitbucket.jenkins.internal.applink.oauth.serviceprovider.consumer.ServiceProviderConsumerStore;
import com.atlassian.bitbucket.jenkins.internal.applink.oauth.serviceprovider.exception.StoreException;
import com.google.common.annotations.VisibleForTesting;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import java.net.URI;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.spec.EncodedKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.inject.Singleton;
import org.apache.commons.lang3.StringUtils;

@Singleton
public class PersistentServiceProviderConsumerStore
extends AbstractPersistentStore<Consumer>
implements ServiceProviderConsumerStore {
    private static final Logger log = Logger.getLogger(PersistentServiceProviderConsumerStore.class.getName());
    private static final String CONSUMER_STORE_ENTRY_NAME = "oauth-consumer";
    private static final String CONSUMER_STORE_KEY_NAME = "consumer-key";
    private static final String CONSUMER_STORE_VALUE_NAME = "consumer-details";

    public PersistentServiceProviderConsumerStore() {
        super("oauth-consumers.xml", new ConsumerConverter());
    }

    @VisibleForTesting
    PersistentServiceProviderConsumerStore(Map<String, Consumer> consumerMap) {
        this();
        this.entityMap = consumerMap;
    }

    @Override
    public void add(Consumer consumer) {
        Objects.requireNonNull(consumer, "consumer");
        this.load();
        if (this.entityMap.containsKey(consumer.getKey())) {
            log.warning(() -> String.format("Consumer with key '%s' already exists.", consumer.getKey()));
            throw new StoreException("Consumer already exists");
        }
        this.entityMap.put(consumer.getKey(), consumer);
        this.save();
    }

    @Override
    public Optional<Consumer> get(String key) {
        Objects.requireNonNull(key, "key");
        this.load();
        return Optional.ofNullable((Consumer)this.entityMap.get(key));
    }

    @Override
    public Iterable<Consumer> getAll() {
        this.load();
        return this.entityMap.values();
    }

    @Override
    public void delete(String key) {
        Objects.requireNonNull(key, "key");
        this.load();
        if (this.entityMap.remove(key) != null) {
            this.save();
        }
    }

    @Override
    public void update(Consumer consumer) {
        Objects.requireNonNull(consumer, "consumer");
        this.load();
        if (!this.entityMap.containsKey(consumer.getKey())) {
            log.warning(() -> String.format("Consumer with key '%s' does not exists.", consumer.getKey()));
            throw new StoreException("No Consumer with the given key " + consumer.getKey());
        }
        this.entityMap.replace(consumer.getKey(), consumer);
        this.save();
    }

    @Override
    protected Class<Consumer> getEntityClass() {
        return Consumer.class;
    }

    @Override
    protected String getStoreEntryName() {
        return CONSUMER_STORE_ENTRY_NAME;
    }

    @Override
    protected String getStoreKeyName() {
        return CONSUMER_STORE_KEY_NAME;
    }

    @Override
    protected String getStoreValueName() {
        return CONSUMER_STORE_VALUE_NAME;
    }

    private static final class ConsumerConverter
    implements Converter {
        private static final String CONSUMER_KEY = "key";
        private static final String CONSUMER_NAME = "name";
        private static final String CONSUMER_DESCRIPTION = "description";
        private static final String CALLBACK_URL = "callback-url";
        private static final String CONSUMER_SECRET = "secret";
        private static final String PUBLIC_KEY = "public-key";
        private static final String PUBLIC_KEY_ENCODED = "key";
        private static final String PUBLIC_KEY_ALGORITHM = "algorithm";
        private static final String PUBLIC_KEY_FORMAT = "format";
        private static final String SIGNATURE_METHOD = "signature-method";

        private ConsumerConverter() {
        }

        private static HierarchicalStreamWriter addNode(HierarchicalStreamWriter writer, String name, Object value) {
            writer.startNode(name);
            writer.setValue(Objects.toString(value));
            writer.endNode();
            return writer;
        }

        private static PublicKey unmarshalPublicKey(HierarchicalStreamReader reader) {
            PublicKey publicKey;
            EncodedKeySpec keySpec;
            KeyFactory keyFactory;
            String publicKeyEncoded = null;
            String publicKeyAlgorithm = null;
            String publicKeyFormat = null;
            while (reader.hasMoreChildren()) {
                reader.moveDown();
                String nodeName = reader.getNodeName();
                String nodeValue = reader.getValue();
                switch (nodeName) {
                    case "key": {
                        publicKeyEncoded = nodeValue;
                        break;
                    }
                    case "algorithm": {
                        publicKeyAlgorithm = nodeValue;
                        break;
                    }
                    case "format": {
                        publicKeyFormat = nodeValue;
                        break;
                    }
                    default: {
                        log.fine(() -> String.format("Unknown consumer entry node '%s' with value '%s' is ignored.", nodeName, nodeValue));
                    }
                }
                reader.moveUp();
            }
            if (StringUtils.isBlank(publicKeyEncoded) || StringUtils.isBlank(publicKeyAlgorithm) || StringUtils.isBlank(publicKeyFormat)) {
                HashSet<String> missingItems = new HashSet<String>(3);
                if (StringUtils.isBlank(publicKeyEncoded)) {
                    missingItems.add("key");
                }
                if (StringUtils.isBlank(publicKeyAlgorithm)) {
                    missingItems.add(PUBLIC_KEY_ALGORITHM);
                }
                if (StringUtils.isBlank(publicKeyFormat)) {
                    missingItems.add(PUBLIC_KEY_FORMAT);
                }
                log.warning(() -> String.format("Failed to unmarshal OAuth consumer: '%s' node is missing valid values for the following child nodes: [%s]", PUBLIC_KEY, String.join((CharSequence)",", missingItems)));
                throw new StoreException("Failed to unmarshal OAuth consumer");
            }
            try {
                keyFactory = KeyFactory.getInstance(publicKeyAlgorithm);
            }
            catch (NoSuchAlgorithmException e) {
                log.log(Level.WARNING, "Failed to unmarshal OAuth consumer", e);
                throw new StoreException("Failed to unmarshal OAuth consumer", e);
            }
            byte[] publicKeyDecoded = Base64.getDecoder().decode(publicKeyEncoded);
            if ("X.509".equals(publicKeyFormat)) {
                keySpec = new X509EncodedKeySpec(publicKeyDecoded);
            } else if ("PKCS#8".equals(publicKeyFormat)) {
                keySpec = new PKCS8EncodedKeySpec(publicKeyDecoded);
            } else {
                log.warning("Unknown public key format: " + publicKeyFormat);
                throw new StoreException("Failed to unmarshal OAuth consumer");
            }
            try {
                publicKey = keyFactory.generatePublic(keySpec);
            }
            catch (InvalidKeySpecException e) {
                log.log(Level.WARNING, "Failed to unmarshal OAuth consumer", e);
                throw new StoreException("Failed to unmarshal OAuth consumer", e);
            }
            return publicKey;
        }

        private static void addPublicKeyNode(HierarchicalStreamWriter writer, PublicKey pubKey) {
            writer.startNode(PUBLIC_KEY);
            ConsumerConverter.addNode(writer, "key", Base64.getEncoder().encodeToString(pubKey.getEncoded()));
            ConsumerConverter.addNode(writer, PUBLIC_KEY_ALGORITHM, pubKey.getAlgorithm());
            ConsumerConverter.addNode(writer, PUBLIC_KEY_FORMAT, pubKey.getFormat());
            writer.endNode();
        }

        public boolean canConvert(Class type) {
            return Consumer.class == type;
        }

        public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
            if (!(source instanceof Consumer)) {
                log.severe(() -> String.format("Source must be of type '%s', but it's ", Consumer.class.getName()) + (source != null ? String.format("of type '%s' instead", source.getClass().getName()) : "null"));
                throw new StoreException("Failed to unmarshal OAuth consumer");
            }
            Consumer consumer = (Consumer)source;
            ConsumerConverter.addNode(writer, "key", consumer.getKey());
            ConsumerConverter.addNode(writer, CONSUMER_NAME, consumer.getName());
            consumer.getDescription().ifPresent(description -> ConsumerConverter.addNode(writer, CONSUMER_DESCRIPTION, description));
            consumer.getCallback().ifPresent(callback -> ConsumerConverter.addNode(writer, CALLBACK_URL, callback));
            ConsumerConverter.addNode(writer, SIGNATURE_METHOD, (Object)consumer.getSignatureMethod());
            consumer.getConsumerSecret().ifPresent(secret -> ConsumerConverter.addNode(writer, CONSUMER_SECRET, secret));
            consumer.getPublicKey().ifPresent(pubKey -> ConsumerConverter.addPublicKeyNode(writer, pubKey));
        }

        public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
            String consumerKey = null;
            String consumerName = null;
            String consumerDescription = null;
            String callbackUrl = null;
            String consumerSecret = null;
            String signatureMethod = null;
            PublicKey publicKey = null;
            while (reader.hasMoreChildren()) {
                reader.moveDown();
                String nodeName = reader.getNodeName();
                String nodeValue = reader.getValue();
                switch (nodeName) {
                    case "key": {
                        consumerKey = nodeValue;
                        break;
                    }
                    case "name": {
                        consumerName = nodeValue;
                        break;
                    }
                    case "description": {
                        consumerDescription = nodeValue;
                        break;
                    }
                    case "callback-url": {
                        callbackUrl = nodeValue;
                        break;
                    }
                    case "secret": {
                        consumerSecret = nodeValue;
                        break;
                    }
                    case "signature-method": {
                        signatureMethod = nodeValue;
                        break;
                    }
                    case "public-key": {
                        publicKey = ConsumerConverter.unmarshalPublicKey(reader);
                        break;
                    }
                    default: {
                        log.fine(() -> String.format("Unknown token entry node '%s' with value '%s' is ignored.", nodeName, nodeValue));
                    }
                }
                reader.moveUp();
            }
            Consumer.Builder builder = new Consumer.Builder(consumerKey);
            if (StringUtils.isNotBlank(callbackUrl)) {
                builder.callback(URI.create(callbackUrl));
            }
            if (StringUtils.isNotBlank(consumerSecret)) {
                builder.consumerSecret(consumerSecret);
            }
            if (StringUtils.isNotBlank(signatureMethod)) {
                builder.signatureMethod(Consumer.SignatureMethod.valueOf(signatureMethod));
            }
            if (StringUtils.isNotBlank(consumerName)) {
                builder.name(consumerName);
            }
            if (StringUtils.isNotBlank((CharSequence)consumerDescription)) {
                builder.description(consumerDescription);
            }
            if (publicKey != null) {
                builder.publicKey(publicKey);
            }
            return builder.build();
        }
    }
}

